<?php

namespace App\Traits;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Models\MediaSize;
use Mtc\MercuryDataModels\Form;

trait ContentMappingTrait
{
    private function mapContentList(Collection $contentList): Collection
    {
        return $contentList
            ->filter(fn($content) => $this->isViewableContent($content))
            ->map(fn(Content $content) => $this->isGlobalContent($content)
                ? $this->mapGlobalContent($content)
                : $this->mapContentElement($content));
    }

    private function isViewableContent($content): bool
    {
        if (!empty($content->hidden)) {
            return false;
        }

        if ($this->isGlobalContent($content)) {
            if (
                !empty($content->globalContent->available_from)
                && Carbon::parse($content->globalContent->available_from) > now()
            ) {
                return false;
            }

            if (
                !empty($content->globalContent->available_to)
                && Carbon::parse($content->globalContent->available_to) < now()
            ) {
                return false;
            }
        }

        return true;
    }

    private function isGlobalContent(Content $content): bool
    {
        return !empty($content->global_content_id);
    }

    private function mapGlobalContent(Content $content): array
    {
        return [
            'ui_component' => $content->globalContent?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => [],
        ];
    }

    private function mapContentElement(Content $content): array
    {
        return [
            'ui_component' => $content->contentElement?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => $content->subContent && $this->componentHasUiResource($content) === false
                ? $this->mapContentList($content->subContent)
                : [],
        ];
    }

    private function mapValue(Content $content)
    {
        if ($this->componentHasUiResource($content)) {
            $contentResource = $this->getUiResource($content);
            return new $contentResource($content);
        }

        return match ($content->data['fieldId'] ?? '') {
            'image', 'images' => collect($content->mediaUses)
                ->map(fn($mediaUse) => [
                    'src' => $mediaUse->getUrl('square-tile'),
                    'sizes' => $this->allSizesForUse($content, $mediaUse),
                    'alt' => $mediaUse->alt_text,
                    'caption' => $mediaUse->caption,
                    'title' => $mediaUse->title,
                ]),
            'form-field' => (new FormViewResource(Form::query()
                ->with('sections.questions')
                ->where('id', $content->content)
                ->first()))
                ->setValues([
                    'offer_id' => $this->request->input('offer_id'),
                    'vehicle_id' => $this->request->input('vehicle_id'),
                    'page_id' => $this->request->input('page_id'),
                    'valuation_id' => $this->request->input('valuation_id'),
                ]),
            default => $content->content,
        };
    }

    private function componentHasUiResource(Content $content): bool
    {
        if (!empty($content->global_content_id)) {
            return !empty($content->globalContent->ui_component)
                && class_exists(self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component);
        }

        return !empty($content->contentElement->ui_component)
            && class_exists(self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component);
    }

    private function getUiResource(Content $content): string
    {
        return !empty($content->global_content_id)
            ? self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component
            : self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component;
    }

    private function componentFromField(Content $content)
    {
        return $content->data['uiComponent'] ?? $this->primitiveField($content);
    }

    private function primitiveField(Content $content)
    {
        return match ($content->data['fieldId'] ?? null) {
            'text-content' => 'UiArticleText',
            'text_area' => 'UiArticleText',
            'images' => 'UiArticleBanner',
            'image' => 'UiArticleBanner',
            default => null,
        };
    }

    private function allSizesForUse(Content $content, MediaUse $mediaUse): Collection
    {
        if (!isset($this->mediaSizes)) {
            $this->mediaSizes = MediaSize::query()
                ->where('model', $content->getMorphClass())
                ->get();
        }
        return $this->mediaSizes
            ->keyBy('label')
            ->map(fn(MediaSize $size) => $mediaUse->getUrl($size->label))
            ->put('original', $mediaUse->media->getOriginalUrlAttribute());
    }
}
