<?php

namespace Tests\Feature;

use App\Facades\Feature;
use App\Models\FilterIndex;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Models\FormQuestion;
use Mtc\Crm\Models\FormSection;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\OfferType;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleOfferFinance;
use Tests\TestCase;

class OfferControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testListOffers()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create(['make_id' => $make->id]);
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $make2 = VehicleMake::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $make2->id,
            'filter_type' => 'make',
        ]);
        $model2 = VehicleModel::factory()->create(['make_id' => $make2->id]);
        FilterIndex::factory()->create([
            'filter_id' => $model2->id,
            'filter_type' => 'model',
        ]);

        $type = OfferType::factory()->create();
        VehicleOffer::factory(5)
            ->create([
                'make_id' => $make->id,
                'model_id' => $model->id,
                'published_at' => Carbon::now()->subDays(3),
                'type_id' => $type->id,
            ]);
        VehicleOffer::factory(3)->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'type_id' => $type->id,

        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('offers.index', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('page', $response->json());
        $this->assertArrayHasKey('featured', $response->json());
        $this->assertArrayHasKey('makes', $response->json());
        $this->assertArrayHasKey('models', $response->json());
    }

    public function testSearchOffers()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $type = OfferType::factory()->create();
        VehicleOffer::factory(5)
            ->create([
                'make_id' => $make->id,
                'model_id' => $model->id,
                'published_at' => Carbon::now()->subDays(3),
                'type_id' => $type->id,
            ]);
        VehicleOffer::factory(3);
        $response = $this->asTenant(tenant())
            ->postJson(route('offers.search', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('make', $response->json());
        $this->assertEquals($make->name, $response->json('make'));
        $this->assertArrayHasKey('models', $response->json());
        $this->assertArrayHasKey('offers', $response->json());
    }

    public function testShowOffer()
    {
        Http::fake([
            'https://cdn.imagin.studio/getPaints*' => Http::response(['paintData' => ['paintCombinations' => []]])
        ]);
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);
        $offer = VehicleOffer::factory()->create();
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
    }

    public function testShowOfferWithSeoData()
    {
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'seo' => ['title' => 'foo', 'description' => 'baz bar']
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('seo.title'));
        $this->assertEquals('baz bar', $response->json('seo.description'));

        SeoDefault::query()->create([
            'section' => 'offer',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);

        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertStringContainsString($offer->name, $response->json('seo.title'));
        $this->assertStringContainsString($make->name, $response->json('seo.title'));
    }

    public function testOfferHasForm()
    {
        $form = Form::factory()->create();
        $section = FormSection::factory()->create(['form_id' => $form->id]);
        FormQuestion::factory()->create([
            'form_id' => $form->id,
            'form_section_id' => $section->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);

        $offer = VehicleOffer::factory()->create();

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);

        $offer->update([
            'form_id' => $form->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('forms.enquire'));
        $this->assertEquals($form->id, $response->json('forms.enquire.id'));
    }

    public function testOfferShowsDealershipNumber()
    {
        $dealership1 = Dealership::factory()->create([
            'contact_no' => '123123',
        ]);
        $dealership2 = Dealership::factory()->create([
            'contact_no' => null
        ]);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $offer1 = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership1
        ]);
        $offer2 = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership2
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer1));
        $response->assertStatus(200);
        $this->assertIsArray($response->json('contact_number'));
        $this->assertArrayHasKey('number', $response->json('contact_number'));
        $this->assertArrayHasKey('display', $response->json('contact_number'));
        $this->assertStringContainsString('123123', $response->json('contact_number.number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer2));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));
    }

    public function testOfferShowData()
    {
        $dealership = Dealership::factory()->create();
        $body_style = BodyStyleType::factory()->create();
        $fuel = FuelType::factory()->create();
        $transmission = TransmissionType::factory()->create();

        $make = VehicleMake::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);

        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership,
            'fuel_type_id' => $fuel->id,
            'transmission_id' => $transmission->id,
            'body_style_id' => $body_style->id,
            'mpg' => 123,
            'battery_range' => 456,
            'battery_capacity_kwh' => 789,
            'battery_usable_capacity_kwh' => 321,
            'battery_charge_time' => 654,
            'battery_quick_charge_time' => 987,
            'engine_size_cc' => 1995,
            'co2' => 555,
            'key_features' => [
                'a',
                'b',
                'c'
            ],
            'technical_spec' => [
                'd',
                'e',
            ],
            'standard_spec' => [
                'x',
            ],
        ]);

        $finance = new VehicleOfferFinance();
        $finance->offer_id = $offer->id;
        $finance->monthly_price = 300;
        $finance->save();

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));
        $response->assertStatus(200);

        $this->assertArrayHasKey('name', $response->json());
        $this->assertArrayHasKey('dealership', $response->json());
        $this->assertArrayHasKey('fuel_type', $response->json());
        $this->assertArrayHasKey('transmission_type', $response->json());
        $this->assertArrayHasKey('colour', $response->json());
        $this->assertArrayHasKey('mpg', $response->json());
        $this->assertArrayHasKey('battery_range', $response->json());
        $this->assertArrayHasKey('battery_capacity_kwh', $response->json());
        $this->assertArrayHasKey('battery_usable_capacity_kwh', $response->json());
        $this->assertArrayHasKey('battery_charge_time', $response->json());
        $this->assertArrayHasKey('battery_quick_charge_time', $response->json());
        $this->assertArrayHasKey('engine_size_cc', $response->json());
        $this->assertArrayHasKey('body_style', $response->json());
        $this->assertArrayHasKey('co2', $response->json());
        $this->assertArrayHasKey('key_features', $response->json());
        $this->assertArrayHasKey('technical_spec', $response->json());
        $this->assertArrayHasKey('standard_spec', $response->json());
        $this->assertArrayHasKey('finance', $response->json());

        $this->assertEquals($dealership->name, $response->json('dealership'));
        $this->assertEquals($fuel->name, $response->json('fuel_type'));
        $this->assertEquals($transmission->name, $response->json('transmission_type'));
        $this->assertEquals($offer->mpg, $response->json('mpg'));
        $this->assertEquals($offer->battery_range, $response->json('battery_range'));
        $this->assertEquals($offer->battery_capacity_kwh, $response->json('battery_capacity_kwh'));
        $this->assertEquals($offer->battery_usable_capacity_kwh, $response->json('battery_usable_capacity_kwh'));
        $this->assertEquals($offer->battery_charge_time, $response->json('battery_charge_time'));
        $this->assertEquals($offer->battery_quick_charge_time, $response->json('battery_quick_charge_time'));
        $this->assertEquals($offer->engine_size_cc, $response->json('engine_size_cc'));
        $this->assertEquals($body_style->name, $response->json('body_style'));
        $this->assertEquals($offer->co2, $response->json('co2'));

        $this->assertIsArray($response->json('key_features'));
        $this->assertIsArray($response->json('technical_spec'));
        $this->assertIsArray($response->json('standard_spec'));
        $this->assertIsArray($response->json('finance'));
        $this->assertCount(3, $response->json('key_features'));
        $this->assertCount(2, $response->json('technical_spec'));
        $this->assertCount(1, $response->json('standard_spec'));
        $this->assertCount(1, $response->json('finance'));
        $this->assertEquals(300, $response->json('finance.0.monthly_price'));
    }
}
