<?php

namespace Tests\Feature;

use App\Facades\Feature;
use App\Models\FilterIndex;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Models\FormQuestion;
use Mtc\Crm\Models\FormSection;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;

class NewCarControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testSearch()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);

        NewCar::factory(5)
            ->create([
                'make_id' => $make->id,
                'published_at' => Carbon::now()->subDays(3),
            ]);
        NewCar::factory(3);
        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('make', $response->json());
        $this->assertEquals($make->name, $response->json('make'));
        $this->assertArrayHasKey('models', $response->json());
        $this->assertArrayHasKey('offers', $response->json());
    }

    public function testShowCar()
    {
        Http::fake([
            'https://cdn.imagin.studio/getPaints*' => Http::response(['paintData' => ['paintCombinations' => []]])
        ]);
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);
        $offer = NewCar::factory()->create();
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));

        $response->assertStatus(200);
    }

    public function testShowCarWithSeoData()
    {
        $car = NewCar::factory()->create(['seo' => ['title' => 'foo', 'description' => 'baz bar']]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('seo.title'));
        $this->assertEquals('baz bar', $response->json('seo.description'));

        SeoDefault::query()->create([
            'section' => 'new-car',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);

        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $car = NewCar::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertStringContainsString($car->name, $response->json('seo.title'));
        $this->assertStringContainsString($make->name, $response->json('seo.title'));
    }

    public function testCarHasForm()
    {
        $form = Form::factory()->create();
        $section = FormSection::factory()->create(['form_id' => $form->id]);
        FormQuestion::factory()->create([
            'form_id' => $form->id,
            'form_section_id' => $section->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);

        $car = NewCar::factory()->create();

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);

        $car->update([
            'form_id' => $form->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('forms.enquire'));
        $this->assertEquals($form->id, $response->json('forms.enquire.id'));
    }

    public function testCarShowsDealershipNumber()
    {
        $dealership1 = Dealership::factory()->create([
            'contact_no' => '123123',
        ]);
        $dealership2 = Dealership::factory()->create([
            'contact_no' => null
        ]);
        $offer = NewCar::factory()->create();
        $offer1 = NewCar::factory()->create(['dealership_id' => $dealership1]);
        $offer2 = NewCar::factory()->create(['dealership_id' => $dealership2]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer1));
        $response->assertStatus(200);
        $this->assertIsArray($response->json('contact_number'));
        $this->assertArrayHasKey('number', $response->json('contact_number'));
        $this->assertArrayHasKey('display', $response->json('contact_number'));
        $this->assertStringContainsString('123123', $response->json('contact_number.number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer2));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));
    }
}
