<?php

namespace App\Http\Controllers;

use App\Http\Resources\GlobalContentResource;
use App\Http\Resources\ListingResource;
use App\Http\Resources\PageResource;
use App\Traits\CacheObject;
use App\PageRepository;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Session;
use Mtc\ContentManager\Models\Page;
use Mtc\MercuryDataModels\GlobalContent;

class PageController extends Controller
{
    use CacheObject;

    /**
     * Show page detail
     *
     * @param Request $request
     * @param string $pageSlug
     * @return PageResource|Response
     */
    public function show(Request $request, string $pageSlug): PageResource|Response
    {
        // Not using binding in route as to allow for Seo Redirect middleware
        /** @var Page $page */
        $page = Page::query()->where('slug', $pageSlug)->firstOrFail();
        if ($page->status !== 'published' && !$this->withAccessKey($request, $page)) {
            return response('Access denied', 401);
        }
        if ($this->restrictedPage($page, $request->input())) {
            return response('Access denied', 401);
        }
        return new PageResource($page);
    }

    public function globalContent($slug)
    {
        // Cache disabled as causes issues with returning empty content
        //return $this->cache("global-content-$slug", 60, function () use ($slug) {
        return new GlobalContentResource(GlobalContent::query()
            ->where('slug', $slug)
            ->firstOrFail());
        //});
    }

    private function restrictedPage(Page $page, array $input): bool
    {
        if (empty($page->password)) {
            return false;
        }

        $submitted_password = $input['password'] ?? Session::get('page-auth-password');
        if ($submitted_password === $page->password) {
            Session::put('page-auth-password', $submitted_password);
        }

        return $submitted_password !== $page->password;
    }

    public function paginatedListing(Request $request, PageRepository $pageRepository, $type = null): array
    {
        $categories = null;
        $tags = null;
        $featured = null;
        $page = Page::query()->where('slug', $type)->active()->first();

        $listing = $pageRepository->listing(
            $type,
            $request->input('perPage') ?? 15,
            'published_at',
            $request->input(),
            true
        );

        if ($request->input('withCategories')) {
            $categories = $pageRepository->getCategories($type);
        }

        if ($request->input('withTags')) {
            $tags = $pageRepository->getTags($type);
        }

        if ($request->input('withFeatured')) {
            $featured = $pageRepository->featured($type, 3, $request->input());
        }

        return [
            'listing' => new ListingResource($listing, $categories, $tags, $featured),
            'page' => $page ? new PageResource($page) : null,
        ];
    }

    private function withAccessKey(Request $request, Page $page): bool
    {
        return $request->input('a') === base64_encode($page?->id . '-' . $page?->slug);
    }
}
