<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\ContentManager\Factories\MediaSizeFactory;
use Mtc\ContentManager\Models\MediaSize;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DealershipDepartment;
use Mtc\MercuryDataModels\Media;
use Mtc\MercuryDataModels\MediaUse;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoData;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;

class DealershipControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testDealershipIndex()
    {
        Dealership::factory(3)->create();
        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('page', $response->json());
        $this->assertArrayHasKey('dealerships', $response->json());
    }

    public function testDealershipShow()
    {
        $dealership = Dealership::factory()->create();
        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.show', $dealership));

        $response->assertStatus(200);
    }

    public function testMedia()
    {
        MediaSize::factory(3)->create([
            'model' => 'dealership',
        ]);

        $dealership = Dealership::factory()->create();
        $media_item = Media::factory()->create();
        MediaUse::factory()->create([
            'media_id' => $media_item->id,
            'owner_type' => 'dealership',
            'owner_id' => $dealership->id
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.show', $dealership));

        $response->assertStatus(200);

        $this->assertCount(1, $response->json('images'));
        $this->assertCount(3, $response->json('images.0.sizes'));

        foreach ($response->json('images.0.sizes') as $size_name => $size_path) {
            $this->assertStringContainsString($media_item->src, $size_path);
        }
    }

    public function testDealershipFindUsSeo()
    {
        $defaults = [
            [
                'section' => 'page',
                'title' => '{{TITLE}} | {{SITE_NAME}} (page)',
                'description' => '{{CONTENT_EXCERPT}} (page)',
            ],
            [
                'section' => 'dealership',
                'title' => '{{TITLE}} | {{SITE_NAME}}',
                'description' => '(dealership)',
            ],
            [
                'section' => 'offer',
                'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} (offer)',
                'description' => '{{CONTENT_EXCERPT}} (offer)',
            ],
            [
                'section' => 'filter',
                'title' => '{{BODY_STYLE}} {{MAKE}} {{MODEL}} Used Vehicle Offers | Page {{PAGE}} | {{SITE_NAME}}',
                'description' => '',
            ],
            [
                'section' => 'offers',
                'title' => '{{BODY_STYLE}} {{MAKE}} {{MODEL}} New Vehicle Offers | {{SITE_NAME}}',
                'description' => '',
            ],
            [
                'section' => 'vehicle',
                'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} (vehicle)',
                'description' => '{{CONTENT_EXCERPT}} (vehicle)',
            ],
        ];

        collect($defaults)
            ->each(fn($default) => SeoDefault::query()->updateOrCreate(['section' => $default['section']], $default));

        $dealership = Dealership::factory()->create();

        SeoData::factory()->create([
            'path' => '/dealership/' . $dealership->slug,
            'title' => 'seo_title',
            'description' => 'seo_description',
        ]);

        $response = $this->asTenant(tenant())
            ->withHeader('x-path', '/my-franchise/find-us/' . $dealership->slug)
            ->getJson(route('dealerships.show', $dealership));

        $response->assertStatus(200);

        // assert that we get the expected SEO data
        $this->assertArrayHasKey('seo', $response->json());
        $this->assertEquals('seo_title', $response->json('seo.title'));
        $this->assertEquals('seo_description', $response->json('seo.description'));
    }

    public function testDealershipListDepartments()
    {
        $dealership = Dealership::factory()->create();

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'foo',
            'open_times' => [],
        ]);

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'bar',
            'open_times' => [],
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('dealerships', $response->json());
        $this->assertCount(1, $response->json('dealerships'));
        $this->assertArrayHasKey('departments', $response->json('dealerships.0'));
        $this->assertCount(2, $response->json('dealerships.0.departments'));

        collect($response->json('dealerships.0.departments'))->each(function ($department) {
            $this->assertContains(
                $department['name'],
                [
                    'foo',
                    'bar'
                ],
            );
        });
    }

    public function testDealershipViewDepartments()
    {
        $dealership = Dealership::factory()->create();

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'foo',
            'open_times' => [],
        ]);

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'bar',
            'open_times' => [],
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.show', $dealership));

        $response->assertStatus(200);

        $this->assertArrayHasKey('departments', $response->json());
        $this->assertCount(2, $response->json('departments'));

        collect($response->json('departments'))->each(function ($department) {
            $this->assertContains(
                $department['name'],
                [
                    'foo',
                    'bar'
                ],
            );
        });
    }

    public function testDealershipsWithoutDepartmentsOpenTimesFromSettings()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'dealership-secondary-open-close-time-name',
            'value' => 'baz',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-open-time',
            'value' => '00:00',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-close-time',
            'value' => '23:59',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-open-days',
            'value' => 'all',
            'type' => 'string',
        ]);


        Dealership::factory()->create();

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('dealerships', $response->json());
        $this->assertCount(1, $response->json('dealerships'));
        $this->assertCount(0, $response->json('dealerships.0.departments'));

        $this->assertEquals('00:00', $response->json('dealerships.0.open_times.0.open'));
        $this->assertEquals('23:59', $response->json('dealerships.0.open_times.0.close'));
        $this->assertNull($response->json('dealerships.0.alt_open_times.0.open'));
        $this->assertNull($response->json('dealerships.0.alt_open_times.0.close'));
        $this->assertNull($response->json('dealerships.0.second_alt_open_times.0.open'));
        $this->assertNull($response->json('dealerships.0.second_alt_open_times.0.close'));
    }

    public function testDealershipsWithoutDepartmentsOpenTimesFromTableData()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'dealership-secondary-open-close-time-name',
            'value' => 'baz',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-open-time',
            'value' => '00:00',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-close-time',
            'value' => '23:59',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-dealership-open-days',
            'value' => 'all',
            'type' => 'string',
        ]);


        Dealership::factory()->create([
            'open_times' => $this->getOpenTimes('11:11', '22:22')
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('dealerships', $response->json());
        $this->assertCount(1, $response->json('dealerships'));
        $this->assertCount(0, $response->json('dealerships.0.departments'));

        $this->assertEquals('11:11', $response->json('dealerships.0.open_times.0.open'));
        $this->assertEquals('22:22', $response->json('dealerships.0.open_times.0.close'));
        $this->assertNull($response->json('dealerships.0.alt_open_times.0.open'));
        $this->assertNull($response->json('dealerships.0.alt_open_times.0.close'));
        $this->assertNull($response->json('dealerships.0.second_alt_open_times.0.open'));
        $this->assertNull($response->json('dealerships.0.second_alt_open_times.0.close'));
    }

    public function testDealershipDepartmentOpenTimes()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'dealership-secondary-open-close-time-name',
            'value' => 'baz',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'dealership-tertiary-open-close-time-name',
            'value' => 'bar',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create();

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'foo',
            'open_times' => $this->getOpenTimes(),
            'is_primary' => true,
        ]);

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'baz',
            'open_times' => $this->getOpenTimes('05:00', '22:00'),
            'is_primary' => false,
        ]);

        DealershipDepartment::query()->create([
            'dealership_id' => $dealership->id,
            'name' => 'bar',
            'open_times' => $this->getOpenTimes('11:00', '17:00'),
            'is_primary' => false,
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('dealerships.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('dealerships', $response->json());
        $this->assertCount(1, $response->json('dealerships'));
        $this->assertCount(3, $response->json('dealerships.0.departments'));

        // test dealership open times
        $this->assertEquals('00:00', $response->json('dealerships.0.open_times.0.open'));
        $this->assertEquals('23:59', $response->json('dealerships.0.open_times.0.close'));
        $this->assertEquals('05:00', $response->json('dealerships.0.alt_open_times.0.open'));
        $this->assertEquals('22:00', $response->json('dealerships.0.alt_open_times.0.close'));
        $this->assertEquals('11:00', $response->json('dealerships.0.second_alt_open_times.0.open'));
        $this->assertEquals('17:00', $response->json('dealerships.0.second_alt_open_times.0.close'));

        // test dealership departments
        $this->assertEquals('00:00', $response->json('dealerships.0.departments.foo.open_times.0.open'));
        $this->assertEquals('23:59', $response->json('dealerships.0.departments.foo.open_times.0.close'));
        $this->assertEquals('05:00', $response->json('dealerships.0.departments.baz.open_times.0.open'));
        $this->assertEquals('22:00', $response->json('dealerships.0.departments.baz.open_times.0.close'));
        $this->assertEquals('11:00', $response->json('dealerships.0.departments.bar.open_times.0.open'));
        $this->assertEquals('17:00', $response->json('dealerships.0.departments.bar.open_times.0.close'));
    }

    private function getOpenTimes(string $open = '00:00', $close = '23:59'): array
    {
        return [
            [
                "dayOfWeek" => 1,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 2,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 3,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 4,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 5,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 6,
                "open" => $open,
                "close" => $close
            ],
            [
                "dayOfWeek" => 7,
                "open" => $open,
                "close" => $close
            ]
        ];
    }
}
