<?php

namespace App;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\FilterObject;
use Mtc\Filter\Contracts\FilterSeoContract;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\SeoDefault;

class Filter extends \Mtc\Filter\Filter
{
    protected $applyFiltersToResults = true;

    private ?SeoDefault $seo_default;

    /**
     * Filter constructor
     *
     * @param Request $request
     * @param FilterObject $product_handler
     * @param FilterSeoContract $seo
     */
    public function __construct(Request $request, FilterObject $product_handler, FilterSeoContract $seo)
    {
        $this->request = $request;
        $this->seo = $seo;
        $this->config = Config::get('filter');
        $this->product_handler = $product_handler;
        $this->active_sort_option_name = $this->config['default_sort_choice'];
    }

    /**
     * Find the id of the slug
     *
     * @param string $model_type
     * @param $slug
     * @return int
     */
    public static function matchSlugToId(string $model_type, $slug)
    {
        return FilterIndex::query()
            ->where('filter_type', $model_type)
            ->where('slug', $slug)
            ->firstOrFail()
            ->filter_id;
    }

    /**
     * Find the id of the slug
     *
     * @param string $model_type
     * @param $slug
     * @return Model
     */
    public static function matchSlugToRecord(string $model_type, $slug)
    {
        return FilterIndex::query()
            ->where('filter_type', $model_type)
            ->where('slug', $slug)
            ->firstOrFail()
            ->filter;
    }

    /**
     * Parse Request to selections and filter data
     *
     * @return array
     */
    public function parseRequest(): array
    {
        $this->decodeRequest();

        return [
            'seo' => $this->getPageSeoData(),
            'selections' => $this->getSelectionList(),
            'sort_by' => $this->active_sort_option_name,
            'base_url' => url($this->config['url_entrypoint']),
            'page' => $this->request->input('page', 1),
        ];
    }

    /**
     * Handle Ajax request, perform filtering
     *
     * @return array
     */
    public function handle(): array
    {
        $this->request = request();
        if ($this->request->has('slug')) {
            $this->filter_url_elements = explode('/', $this->request->input('slug'));
            $this->matchSortFromUrlElements();
            $this->matchSelectedFilters();
            $this->matchCustomSelections();
            $this->checkForSearchTerm();
        } else {
            $this->selections = $this->groupSelectionsByType($this->request->input('selections') ?? []);
        }
        $this->run();

        return [
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'sort_options' => $this->getSortOptions(),
            'sort_by' => $this->active_sort_option_name,
            'selections' => $this->getSelectionList(),
            'url' => $this->getPageUrl(),
            'seo' => [
                'title' => trim($this->seoTitle()),
                'description' => trim($this->seoDescription()),
            ]
        ];
    }

    public function handleWidget()
    {
        $this->request = request();
        $this->applyFiltersToResults = false;
        $this->selections = $this->groupSelectionsByType($this->request->input('selections') ?? []);
        $this->run();
        return [
            'result_count' => $this->getResultCount(),
            'filters' => $this->getFilterResults(true),
            'url' => $this->getPageUrl(),
        ];
    }

    /**
     * Get enabled filters
     *
     * @return Collection
     */
    public function getFilters(): Collection
    {
        if (empty($this->filters)) {
            $this->filters = collect($this->config['filters'])
                ->map(fn($filter_class) => App::make($filter_class))
                ->filter(fn($filter) => $filter instanceof IsFilter)
                ->filter(fn($filter, $name) => Settings::get("automotive-vehicle-filters-$name") === true)
                ->sortBy(fn($filter, $name) => Settings::getOrder("automotive-vehicle-filters-$name"));
        }

        return $this->filters;
    }

    /**
     * Fetch the list of supported sort options
     * Returned in format of slug => display name
     *
     * @return array
     */
    public function getSortOptions(): array
    {
        return collect($this->config['sort_options'])
            ->filter(fn($filter, $filter_name) => Settings::get("automotive-vehicle-sorting-$filter_name") === true)
            ->sortBy(fn($filter, $name) => Settings::getOrder("automotive-vehicle-sorting-$name"))
            ->map(fn($sort_class, $sort_name) => [
                'code' => $sort_name,
                'label' => __("filter::filter.sort_options.$sort_name")
            ])
            ->values()
            ->toArray();
    }

    /**
     * Get the selections as a flat list
     *
     * @return Collection
     */
    public function getSelectionList(): Collection
    {
        if ($this->index === null) {
            $this->getFilterIndexForSelections();
        }

        return collect($this->selections)
            ->flatMap(fn($filter_group, $filter_type) => collect($filter_group)
                ->map(fn($value) => [
                    'type' => $filter_type,
                    'value' => $value,
                    'name' => $this->selectionName($filter_type, $value)
                ]));
    }

    /**
     * Build the URL of the page that has been viewed
     *
     * @param bool $absolute
     * @return string
     */
    public function getPageUrl(bool $absolute = true): string
    {
        $append = '';
        $customSort = $this->active_sort_option_name !== $this->config['default_sort_choice'];
        $params = array_filter([
            'page' => $this->request->input('page') > 1 ? $this->request->input('page') : null,
            'sort_by' => $customSort ? $this->active_sort_option_name : null,
        ]);
        if (!empty($params)) {
            $append = '?' . http_build_query($params);
        }
        return $this->filterElementsToSlugs($this->selections)->join('/') . $append;
    }


    /**
     * Get the selection name from index
     *
     * @param string $type
     * @param $value
     * @return string|null
     */
    protected function selectionName(string $type, $value): ?string
    {
        $filterInstance = App::make($this->config['filters'][$type]);
        $customPatternFilter = $filterInstance instanceof CustomPatternFilter;

        if ($customPatternFilter === false) {
            return $this->selectionNameInIndex($type, $value);
        }

        return $filterInstance->getSelectionName($value);
    }


    /**
     * Get the selection name from index
     *
     * @param string $type
     * @param $value
     * @return string|null
     */
    protected function selectionNameInIndex(string $type, $value): ?string
    {
        return $this->index
            ->where('filter_type', $type)
            ->firstWhere('filter_id', $value)
            ?->name;
    }

    /**
     * Create a list of all filter elements (selections, ordering, terms etc.) in one list of slugs
     *
     * @param array $selections
     * @return Collection
     */
    protected function filterElementsToSlugs(array $selections): Collection
    {
        $all = collect([]);
        collect($selections)
            ->sortBy(fn($selection, $type) => array_search($type, array_keys($this->config['filters']), true))
            ->each(function ($type_selections, $type) use ($all) {
                $filter = $this->getFilter($type);
                // Unsupported selection
                if ($filter === null) {
                    return;
                }

                collect($type_selections)->each(function ($selection) use ($all, $filter, $type) {
                    $slugs = $this->getFilterIndexForSelections()
                        ->groupBy('filter_type')
                        ->map(fn(Collection $group) => $group->keyBy('filter_id'));

                    if ($filter instanceof CustomPatternFilter) {
                        $all->push($filter->createSlug($selection));
                    } elseif (isset($slugs[$type][$selection]['slug'])) {
                        $all->push($slugs[$type][$selection]['slug']);
                    }
                });
            });

        return $all;
    }

    /**
     * Retrieve filter results based on selections
     *
     * @param bool $prependAny
     * @return array
     */
    protected function getFilterResults(bool $prependAny = false): array
    {
        $input = $this->request->input();
        return $this->getFilters()
            ->when(
                !empty($input['only_filters']),
                fn ($all) => $all->filter(fn ($filter, $name) => in_array($name, $input['only_filters'] ?? [], true))
            )
            ->map(fn(IsFilter $filter, $name) => $filter->format($this->retrieveSingleFilterResults($filter, $name)))
            ->filter()
            ->map(function (array $filterGroup, $type) use ($prependAny) {
                $filterGroup['results'] = collect($filterGroup['results'])
                    ->map(function ($entry) use ($type) {
                        if (is_array($entry)) {
                            $valueField = $type === 'price' ? 'value' : 'id';
                            $entry['selected'] = isset($this->selections[$type])
                                && in_array($entry[$valueField], $this->selections[$type]);
                        }
                        return $entry;
                    })
                    ->when($prependAny, fn ($group) => $group->prepend([
                        'id' => 0,
                        'name' => __('labels.any'),
                    ]));
                return $filterGroup;
            })
            ->toArray();
    }

    /**
     * Get (product) results.
     * Sets the orderBy for the main query
     * Executes the query to retrieve data
     * Formats data to defined response format
     *
     * @return int
     */
    protected function getResultCount(): int
    {
        return $this->product_handler->getResultCount(clone $this->query);
    }

    /**
     * Apply filters for other filters.
     * This applies the same  filters as result filtering except for current active filter.
     * It also adds product filter condition
     *
     * @param Builder $query
     * @param string $active_filter
     */
    protected function applyForFilters(Builder $query, string $active_filter): void
    {
        if ($this->applyFiltersToResults) {
            $this->getFilters()
                ->reject(
                    fn(IsFilter $filter, string $type) => empty($this->selections[$type]) || $active_filter === $type
                )
                ->each(fn(IsFilter $filter, string $type) => $filter->applyFilter($query, $this->selections[$type]));
        } elseif ($active_filter === 'model') {
            $this->getFilters()
                ->filter(fn(IsFilter $filter, string $type) => !empty($this->selections[$type]) && 'make' === $type)
                ->each(fn(IsFilter $filter, string $type) => $filter->applyFilter($query, $this->selections[$type]));
        }

        $this->product_handler->applyFilter($query);
    }

    /**
     * Group selections by type
     * @param array $selections
     * @param string $group_by
     * @return array
     */
    protected function groupSelectionsByType(array $selections = [], string $group_by = 'type'): array
    {
        return collect($selections)
            ->filter(fn ($selection) => !empty($selection['value']))
            ->groupBy($group_by)
            ->map(fn($group) => collect($group)->pluck('value'))
            ->toArray();
    }

    private function seoTitle(): string
    {
        $default = $this->seoDefault();

        return $this->replaceSeoTags([
            '{{MAKE}}' => $this->replaceSelectionByType('make'),
            '{{MODEL}}' => $this->replaceSelectionByType('model'),
            '{{FUEL_TYPE}}' => $this->replaceSelectionByType('fuel_type'),
            '{{BODY_STYLE}}' => $this->replaceSelectionByType('body_type'),
            '{{TRANSMISSION}}' => $this->replaceSelectionByType('transmission'),
            '{{PAGE}}' => $this->getCurrentPage(),
            '{{SITE_NAME}}' => Settings::get('app-name'),

        ], $default->title ?? '');
    }

    private function seoDescription(): string
    {
        $default = $this->seoDefault();

        return $this->replaceSeoTags([
            '{{MAKE}}' => $this->replaceSelectionByType('make'),
            '{{MODEL}}' => $this->replaceSelectionByType('model'),
            '{{FUEL_TYPE}}' => $this->replaceSelectionByType('fuel_type'),
            '{{BODY_STYLE}}' => $this->replaceSelectionByType('body_type'),
            '{{TRANSMISSION}}' => $this->replaceSelectionByType('transmission'),
            '{{PAGE}}' => $this->getCurrentPage(),
            '{{SITE_NAME}}' => Settings::get('app-name'),

        ], $default->description ?? '');
    }

    private function seoDefault(): ?SeoDefault
    {
        if (!isset($this->seo_default)) {
            $this->seo_default = SeoDefault::query()->where('section', 'filter')->first();
        }
        return $this->seo_default;
    }

    private function replaceSelectionByType(string $type): string
    {
        if (empty($this->selections[$type]) || count($this->selections[$type]) > 1) {
            return '';
        }

        return $this->selectionNameInIndex($type, $this->selections[$type][0]);
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }
}
