<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Form;

class PageResource extends JsonResource
{
    use FranchiseData;

    private const UI_RESOURCE_PREFIX = 'App\\Http\\Resources\\';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        if ($request->filled('v')) {
            $this->resource->data = $this->previewVersion($request->input('v'));
        }
        if (empty($this->resource->data)) {
            $this->resource->load([
                'rootLevelContent.mediaUses.media',
                'rootLevelContent.subContent.mediaUses.media',
                'rootLevelContent.subContent.subContent.mediaUses.media',
                'rootLevelContent.subContent.subContent.subContent.mediaUses.media',
                'rootLevelContent.contentElement',
                'rootLevelContent.globalContent.contentElement',
                'rootLevelContent.subContent.contentElement',
                'rootLevelContent.subContent.globalContent.contentElement',
                'rootLevelContent.subContent.subContent.contentElement',
                'rootLevelContent.subContent.subContent.globalContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.contentElement',
                'rootLevelContent.subContent.subContent.subContent.globalContent.contentElement',
            ]);
            $this->resource->data = $this->mapContentList($this->resource->rootLevelContent);
        }

        $this->resource->setHidden([
            'text',
        ]);

        if ($this->resource->slug === 'valuation') {
            $this->resource->enquiry = $this->valuationEnquiry();
        }

        $resource = $this->resource->toArray();
        $resource['isFranchise'] = $this->isFranchise();
        $resource['franchise'] = $this->franchiseData();
        $resource['seo'] = $this->fillSeo();
        $resource['published_at'] = $this->resource->published_at?->format('F j, Y');
        unset($resource['root_level_content']);
        return $resource;
    }

    private function previewVersion(mixed $versionId): ?Collection
    {
        /** @var VersionModel $version */
        $version = $this->resource->versions()
            ->with([
                'content.mediaUses.media',
                'content.subContent.mediaUses.media',
                'content.subContent.subContent.mediaUses.media',
                'content.subContent.subContent.subContent.mediaUses.media',
                'content.contentElement',
                'content.globalContent.contentElement',
                'content.subContent.contentElement',
                'content.subContent.globalContent.contentElement',
                'content.subContent.subContent.contentElement',
                'content.subContent.subContent.globalContent.contentElement',
                'content.subContent.subContent.subContent.contentElement',
                'content.subContent.subContent.subContent.globalContent.contentElement',
            ])
            ->where('uuid', $versionId)
            ->first();

        return $version ? $this->mapContentList($version->content) : null;
    }

    private function mapContentList(Collection $contentList): Collection
    {
        return $contentList->map(
            fn(Content $content) => $this->isGlobalContent($content)
                ? $this->mapGlobalContent($content)
                : $this->mapContentElement($content)
        );
    }

    private function isGlobalContent(Content $content): bool
    {
        return !empty($content->global_content_id);
    }

    private function mapGlobalContent(Content $content): array
    {
        return [
            'ui_component' => $content->globalContent?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => [],

        ];
    }

    private function mapContentElement(Content $content): array
    {
        return [
            'ui_component' => $content->contentElement?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => $content->subContent && $this->componentHasUiResource($content) === false
                ? $this->mapContentList($content->subContent)
                : [],
        ];
    }


    private function mapValue(Content $content)
    {
        if ($this->componentHasUiResource($content)) {
            $contentResource = $this->getUiResource($content);
            return new $contentResource($content);
        }

        if (in_array($content->data['fieldId'] ?? '', ['image', 'images'])) {
            return collect($content->mediaUses)
                ->map(fn($mediaUse) => [
                    'src' => $mediaUse->getUrl('2560x650'), // TODO correct image size
                    'alt' => $mediaUse->alt_text,
                    'caption' => $mediaUse->caption,
                    'title' => $mediaUse->title,
                ]);
        }

        return $content->content;
    }

    private function componentHasUiResource(Content $content): bool
    {
        if (!empty($content->global_content_id)) {
            return !empty($content->globalContent->ui_component)
                && class_exists(self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component);
        }

        return !empty($content->contentElement->ui_component)
            && class_exists(self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component);
    }

    private function getUiResource(Content $content): string
    {
        return
            !empty($content->global_content_id)
                ? self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component
                : self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component;
    }

    private function componentFromField(Content $content)
    {
        return $content->data['uiComponent'] ?? $this->primitiveField($content);
    }

    private function primitiveField(Content $content)
    {
        return match ($content->data['fieldId'] ?? null) {
            'text-content' => 'UiArticleText',
            'text_area' => 'UiArticleText',
            'images' => 'UiArticleBanner',
            'image' => 'UiArticleBanner',
            default => null,
        };
    }

    private function fillSeo(): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $default = SeoDefault::query()->where('section', 'page')->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{CONTENT_EXCERPT}}' => json_encode($this->resource->text), // TODO: This needs actual content

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }

    private function valuationEnquiry(): ?array
    {
        $form = Form::query()
            ->with('sections.questions')
            ->whereHas('type', fn($query) => $query->where('name', 'Valuation'))
            ->first();
        return [
            'type' => 'form',
            'title' => 'Vehicle Valuation',
            'details' => $form
                ? new FormViewResource($form)
                : [],
        ];
    }
}
