<?php

namespace App\Providers;

use App\Facades\Settings;
use App\Modules\Stock\Motordat;
use App\Newsletter\CampaignMonitor;
use App\Newsletter\MailChimp;
use App\SettingRepository;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\Eloquent\Relations\Relation;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DrivetrainType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        if ($this->app->runningUnitTests()) {
            $this->setMigrationLocation();
        }
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        $this->bindNewsletterDrivers();
        $this->app->bind('site-settings', SettingRepository::class);
        $this->app->bind('motordat', Motordat::class);

        Relation::morphMap([
            'user' => \config('auth.providers.users.model'),
            'dealership' => Dealership::class,
            'vehicle' => Vehicle::class,
            'offer' => VehicleOffer::class,
            'make' => VehicleMake::class,
            'model' => VehicleModel::class,
            'fuel_type' => FuelType::class,
            'transmission' => TransmissionType::class,
            'body_style' => BodyStyleType::class,
            'drivetrain' => DrivetrainType::class,
        ]);

        Model::preventLazyLoading(!$this->app->isProduction());
    }

    /**
     * Set up bindings for newsletter drivers
     *
     * @return void
     */
    private function bindNewsletterDrivers()
    {
        $this->app->bind(MailChimp::class, function () {
            return new MailChimp(new \DrewM\MailChimp\MailChimp(Settings::get('newsletters-mailchimp-api_key')));
        });
        $this->app->bind(CampaignMonitor::class, function () {
            return new CampaignMonitor(new \CS_REST_Subscribers(
                Settings::get('newsletters-campaign-monitor-list_id'),
                ['api_key' => Settings::get('newsletters-campaign-monitor-api_key')]
            ));
        });
    }

    /**
     * Due to the shared data models migrations are moved in mtcmedia/mercury-data-models
     *
     * @return void
     */
    private function setMigrationLocation()
    {
        $this->app->useDatabasePath($this->app->basePath('vendor/mtcmedia/mercury-data-models/database'));
        Config::set(
            'tenancy.migration_parameters.--path',
            $this->app->basePath('vendor/mtcmedia/mercury-data-models/database/migrations/tenant')
        );
    }
}
