<?php

namespace App\Http\Resources;

use App\Events\LoadVehicleDataFields;
use App\Http\Resources\VehicleAction\FinanceFactory;
use App\Http\Resources\VehicleAction\LoadsVehicleActions;
use App\Traits\HasPlaceholderImage;
use App\Traits\RetrievesContent;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\VehicleAttribute;

class VehicleResource extends JsonResource
{
    use LoadsVehicleActions;
    use HasPlaceholderImage;
    use RetrievesContent;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->fuel_type = $this->resource->fuelType?->name;
        $this->resource->transmission = $this->resource->transmission?->name;
        $this->resource->body_style = $this->resource->bodyStyle?->name;
        $this->resource->load([
            'mediaUses.media',
            'specs',
            'features',
        ]);
        $this->resource->media_uses = $this->resource->mediaUses->map(fn($mediaUse) => [
            'title' => $mediaUse->title,
            'alt_text' => $mediaUse->alt_text,
            'description' => $mediaUse->description,
            'caption' => $mediaUse->caption,
            'src_large' => $mediaUse->getUrl('large'),
            'src_medium' => $mediaUse->getUrl('medium'),
            'type' => $mediaUse->media->type
        ]);
        $this->resource->specs = $this->resource->specs->map(fn($spec) => [
            'category' => $spec->category,
            'description' => $spec->description,
            'value' => $spec->value,
        ]);

        $this->resource->features = $this->resource->features->pluck('name');

        if ($this->resource->dealership) {
            $this->resource->dealership->address = collect([
                $this->resource->dealership->address1,
                $this->resource->dealership->address2,
                $this->resource->dealership->city,
                $this->resource->dealership->postcode,
            ])->filter()->implode(', ');
        }

        $this->resource->dealership = $this->resource->dealership?->only([
            'slug',
            'name',
            'contact_no',
            'address',
            'is_open',
            'opens_at',
            'closes_at',
        ]);

        $this->resource->actions = $this->getVehicleActions($this->resource);
        $this->resource->content = $this->getContent();

        $resource = $this->resource->only([
            'actions',
            'bodyStyle',
            'colour',
            'content',
            'dealership',
            'deposit',
            'derivative',
            'drivetrain',
            'engine_size_cc',
            'equipment',
            'features',
            'financeExamples',
            'fuel_type',
            'make',
            'model',
            'odometer_mi',
            'odometer_km',
            'media_uses',
            'monthly_price',
            'original_price',
            'price',
            'registration_number',
            'rrp_price',
            'seats',
            'specs',
            'title',
            'transmission',
        ]);

        if (is_null($resource['media_uses']) || $resource['media_uses']->isEmpty()) {
            $resource['media_uses'] = $this->getPlaceholderImage('vehicle');
        }

        $resource['monthly_cost_type'] = $this->getMonthlyCostType();
        $resource['finance'] = $this->getFinanceData();

        $resource['extra'] = $this->getExtraData();

        $resource['seo'] = $this->fillSeo('vehicle');

        return $resource;
    }

    private function getExtraData()
    {
        return collect(event(new LoadVehicleDataFields($this->resource)))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->map(function ($field, $key) {
                $field['id'] = $key;
                return $field;
            })->values()
            ->merge($this->getCustomAttributes());
    }

    private function getCustomAttributes(): Collection
    {
        $this->resource->load('attributeValues');
        return VehicleAttribute::all()
            ->map(fn(VehicleAttribute $attribute) => $this->allocateAttributeValues($attribute));
    }

    private function getContent(): Collection
    {
        return collect([
            [
                'title' => __('labels.description'),
                'content' => $this->resource->description,
                'buttonLink' => false
            ],
        ])->filter(fn($entry) => !empty($entry['content']));
    }

    private function getMonthlyCostType(): ?string
    {
        if (empty($this->resource->monthly_cost_type)) {
            return null;
        }

        return collect(config('automotive.monthly_cost_types', []))
            ->filter(fn($type) => $type['id'] === $this->resource->monthly_cost_type)
            ->first()['name'] ?? null;
    }

    private function getFinanceData()
    {
        return (new FinanceFactory())->widget($this->resource);
    }
}
