<?php

namespace App\Modules\Purchase;

use App\Facades\Settings;
use App\Modules\Purchase\Http\Requests\CreateInvoiceRequest;
use App\Modules\Purchase\Http\Requests\PaymentOptionRequest;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\CustomerInvoice;
use Mtc\MercuryDataModels\PurchaseAddOn;
use Mtc\MercuryDataModels\Vehicle;

class PurchaseRepository
{
    public function extrasForVehicle(Vehicle $vehicle): Collection
    {
        return PurchaseAddOn::query()
            ->active()
            ->applicableTo($vehicle)
            ->get();
    }

    public function getDeliveryOptions(Vehicle $vehicle): Collection
    {
        return collect();
    }

    public function getPaymentOptions(PaymentOptionRequest $request, Vehicle $vehicle): Collection
    {
        return collect();
    }

    public function createInvoice(CreateInvoiceRequest $request, Vehicle $vehicle): CustomerInvoice
    {
        /** @var CustomerInvoice $invoice */
        $invoice = CustomerInvoice::query()
            ->create([
                'preferred_delivery_date' => $request->input('preferred_delivery_date'),
                'requires_finance' => $request->input('requires_finance'),
                'type' => $request->input('type'),
                'email' => $request->input('address.email'),
                'contact_no' => $request->input('address.contact_no'),
                'due_at' => $this->invoiceDueDate(),
                'vat_applicable' => true,
                'details' => $request->input('details'),
            ]);
        $this->setRecipientAddressToInvoice($request->input('address'), $invoice);
        $this->setSellerAddressToInvoice($invoice);
        $extras = PurchaseAddOn::query()
            ->whereIn('id', $request->input('extras'))
            ->get();
        $this->setItems($vehicle, $extras, $invoice);
        $this->setInvoiceAmount($invoice);
        return $invoice;
    }

    private function setRecipientAddressToInvoice(array $input, CustomerInvoice $invoice): void
    {
        $invoice->addresses()->create([
            'type' => 'recipient',
            'title' => $input['title'] ?? null,
            'company_name' => $input['company_name'] ?? null,
            'first_name' => $input['first_name'],
            'last_name' => $input['last_name'],
            'address1' => $input['address1'] ?? null,
            'address2' => $input['address2'] ?? null,
            'city' => $input['city'] ?? null,
            'county' => $input['county'] ?? null,
            'postcode' => $input['postcode'] ?? null,
            'country' => $input['country'] ?? null,
            'state' => $input['state'] ?? null,
        ]);
    }

    private function setSellerAddressToInvoice(CustomerInvoice $invoice): void
    {
        $invoice->addresses()->create([
            'type' => 'seller',
            'company_name' => Settings::get('app-details-invoice-company-name'),
            'first_name' => Settings::get('app-details-invoice-first-name'),
            'last_name' => Settings::get('app-details-invoice-last-name'),
            'address1' => Settings::get('app-details-invoice-address1'),
            'address2' => Settings::get('app-details-invoice-address2'),
            'city' => Settings::get('app-details-invoice-city'),
            'county' => Settings::get('app-details-invoice-county'),
            'postcode' => Settings::get('app-details-invoice-postcode'),
            'country' => Settings::get('app-details-invoice-country'),
            'state' => Settings::get('app-details-invoice-state'),
        ]);
    }

    private function setItems(Vehicle $vehicle, ?Collection $extras, CustomerInvoice $invoice): void
    {
        $invoice->items()->create([
            'purchasable_id' => $vehicle->id,
            'purchasable_type' => $vehicle->getMorphClass(),
            'name' => $vehicle->title,
            'price' => $this->getVehiclePrice($invoice, $vehicle),
            'price_ex_vat' => $this->getVehiclePrice($invoice, $vehicle),
            'original_price' => $vehicle->original_price,
            'original_price_ex_vat' => $vehicle->original_price,
        ]);

        if ($extras) {
            $extras->each(fn(PurchaseAddOn $addOn) => $invoice->items()->create([
                'purchasable_id' => $addOn->id,
                'purchasable_type' => $addOn->getMorphClass(),
                'name' => $addOn->name,
                'price' => $addOn->price,
                'price_ex_vat' => $addOn->price_ex_vat,
            ]));
        }
    }

    private function getVehiclePrice(CustomerInvoice $invoice, Vehicle $vehicle)
    {
        return match ($invoice->type) {
            'purchase' => $vehicle->price,
            default => $this->vehicleReservePrice($vehicle),
        };
    }

    private function vehicleReservePrice(Vehicle $vehicle): int|float
    {
        return 99;
    }

    private function setInvoiceAmount(CustomerInvoice $invoice)
    {
        $invoice->update([
            'amount' => $invoice->items()->sum('price'),
        ]);
    }

    private function invoiceDueDate()
    {
        return Carbon::now();
    }
}
