<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleFinance;
use Mtc\MercuryDataModels\VehicleView;
use Tests\TestCase;

class VehicleControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFilterLoads()
    {
        Vehicle::factory(10)->create([
            'is_published' => true,
        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('sort_options', $response->json());
        $this->assertArrayHasKey('results', $response->json());
        $this->assertArrayHasKey('filters', $response->json());
        $this->assertEquals(10, $response->json('results.total'));
    }

    public function testViewSingleVehicle()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $bodystyle = BodyStyleType::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
            'body_style_id' => $bodystyle->id
        ]);
        $vehicle->original_price = 123123;
        $vehicle->rrp_price = 456456;
        $vehicle->save();
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('bodyStyle', $response->json('data'));
        $this->assertArrayHasKey('colour', $response->json('data'));
        $this->assertArrayHasKey('content', $response->json('data'));
        $this->assertArrayHasKey('dealership', $response->json('data'));
        $this->assertArrayHasKey('derivative', $response->json('data'));
        $this->assertArrayHasKey('drivetrain', $response->json('data'));
        $this->assertArrayHasKey('engine_size_cc', $response->json('data'));
        $this->assertArrayHasKey('equipment', $response->json('data'));
        $this->assertArrayHasKey('features', $response->json('data'));
        $this->assertArrayHasKey('financeExamples', $response->json('data'));
        $this->assertArrayHasKey('fuel_type', $response->json('data'));
        $this->assertArrayHasKey('make', $response->json('data'));
        $this->assertArrayHasKey('model', $response->json('data'));
        $this->assertArrayHasKey('odometer_km', $response->json('data'));
        $this->assertArrayHasKey('odometer_mi', $response->json('data'));
        $this->assertArrayHasKey('original_price', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
        $this->assertArrayHasKey('registration_number', $response->json('data'));
        $this->assertArrayHasKey('rrp_price', $response->json('data'));
        $this->assertArrayHasKey('seats', $response->json('data'));
        $this->assertArrayHasKey('specs', $response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('transmission', $response->json('data'));

        $this->assertNotNull($response->json('data.bodyStyle'));
        $this->assertNotNull($response->json('data.colour'));
        $this->assertNotNull($response->json('data.content'));
        $this->assertNotNull($response->json('data.dealership'));
        $this->assertNotNull($response->json('data.derivative'));
        $this->assertNotNull($response->json('data.drivetrain'));
        $this->assertNotNull($response->json('data.engine_size_cc'));
        $this->assertNotNull($response->json('data.equipment'));
        $this->assertNotNull($response->json('data.features'));
        $this->assertNotNull($response->json('data.financeExamples'));
        $this->assertNotNull($response->json('data.fuel_type'));
        $this->assertNotNull($response->json('data.make'));
        $this->assertNotNull($response->json('data.model'));
        $this->assertNotNull($response->json('data.odometer_km'));
        $this->assertNotNull($response->json('data.odometer_mi'));
        $this->assertNotNull($response->json('data.original_price'));
        $this->assertNotNull($response->json('data.price'));
        $this->assertNotNull($response->json('data.registration_number'));
        $this->assertNotNull($response->json('data.rrp_price'));
        $this->assertNotNull($response->json('data.seats'));
        $this->assertNotNull($response->json('data.specs'));
        $this->assertNotNull($response->json('data.title'));
        $this->assertNotNull($response->json('data.transmission'));

        $this->assertArrayHasKey('extra', $response->json('data'));

        $has_savings = false;

        foreach ($response->json('data.extra') as $data_field) {
            if ($data_field['id'] == 'savings') {
                $has_savings = true;
            }
        }

        $this->assertTrue($has_savings);
    }

    public function testQuickViewVehicle()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.quick-view', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
        $this->assertArrayHasKey('actions', $response->json('data'));
        $this->assertArrayHasKey('odometer_mi', $response->json('data'));
        $this->assertArrayHasKey('odometer_km', $response->json('data'));
        $this->assertArrayHasKey('fuel_type', $response->json('data'));
        $this->assertArrayHasKey('transmission', $response->json('data'));

        $this->assertTrue(VehicleView::query()->where('vehicle_id', $vehicle->id)->where('quick_view_hits', 1)->exists());
    }

    public function testViewVehicleWithAccessKey()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => false,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));
        $response->assertStatus(403);
        $response = $this->asTenant(tenant())->getJson(
            route('vehicles.show', 'foo-baz-bar') . '?a=' . base64_encode($vehicle->id . '-' . $vehicle->slug)
        );
        $response->assertStatus(200);
    }

    public function testRecentlyViewed()
    {
        $vehicles = Vehicle::factory(3)->create(['is_published' => true]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.recentlyViewed', 'foo-baz-bar') . '?'
                . http_build_query(['slugs' => $vehicles->pluck('slug')->toArray()]));

        $response->assertStatus(200);
        $this->assertArrayHasKey('items', $response->json('recently_viewed'));
        $this->assertCount(3, $response->json('recently_viewed.items'));
    }

    public function testViewWithDefaultFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        VehicleFinance::factory()->create(['vehicle_id' => $vehicle->id]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertEquals('FinanceCalculator', $response->json('data.finance.value'));
        $this->assertArrayHasKey('change_values', $response->json('data.finance.data'));
        $this->assertArrayHasKey('values', $response->json('data.finance.data'));
    }

    public function testViewWithIvendiFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-ivendi-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('IVendiCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithCodeweaversFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-codeweavers-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('CodeweaversCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithEvolutionFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-evolution-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('EvolutionCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }
}
