<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\OfferRepository;
use App\Traits\HasPlaceholderImage;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Country;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\VehicleOfferContent;

class OfferResource extends JsonResource
{
    use HasPlaceholderImage;
    use FranchiseData;
    use PhoneNumberDisplay;

    public static $wrap = '';

    private const UI_RESOURCE_PREFIX = 'App\\Http\\Resources\\';

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->load([
            'mediaUses.media',
            'finance',
            'transmission',
            'fuelType',
            'bodyStyle',
            'drivetrain',
            'form.sections.questions',
            'content.mediaUses.media',
            'content.subContent.mediaUses.media',
            'content.subContent.subContent.mediaUses.media',
            'content.subContent.subContent.subContent.mediaUses.media',
            'content.contentElement',
            'content.globalContent.contentElement',
            'content.subContent.contentElement',
            'content.subContent.globalContent.contentElement',
            'content.subContent.subContent.contentElement',
            'content.subContent.subContent.globalContent.contentElement',
            'content.subContent.subContent.subContent.contentElement',
            'content.subContent.subContent.subContent.globalContent.contentElement',
        ]);
        $this->resource->media_uses = $this->resource->mediaUses->map(fn($mediaUse) => [
            'title' => $mediaUse->title,
            'alt_text' => $mediaUse->alt_text,
            'description' => $mediaUse->description,
            'caption' => $mediaUse->caption,
            'src_large' => $mediaUse->getUrl('900x600'),
            'src_medium' => $mediaUse->getUrl('440x295'),
            'type' => $mediaUse->media->type
        ]);

        $this->resource->make_name = $this->resource->make?->name;
        $this->resource->model_name = $this->resource->model?->name;
        $this->resource->view_stock = false;

        $this->resource->forms = [
            'enquire' => $this->resource->form
                ? (new FormViewResource($this->resource->form))->setValues([ 'offer_id' => $this->resource->id ])
                : null,
        ];

        $this->resource->features = $this->resource->features->pluck('name');

        $resource = $this->resource->only([
            'name',
            'price',
            'make_name',
            'model_name',
            'derivative',
            'content',
            'media_uses',
            'forms',
            'finance',
            'specs',
            'features',
        ]);

        if (is_null($resource['media_uses']) || $resource['media_uses']->isEmpty()) {
            $resource['media_uses'] = $this->getPlaceholderImage('offer');
        }

        $resource['details'] = $this->getDetails();

        $resource['cross_sell'] = (new OfferRepository())->similar($this->resource);
        $resource['contact_number'] = $this->getContactNumber();
        $resource['content'] = $this->getContent($this->resource->content);
        $resource['seo'] = $this->fillSeo();
        $resource['isFranchise'] = $this->isFranchise();
        $resource['franchise'] = $this->franchiseData();
        return $resource;
    }

    private function getContent(Collection $contentList): Collection
    {
        return $contentList->map(
            fn(VehicleOfferContent $content) => $this->isGlobalContent($content)
                ? $this->mapGlobalContent($content)
                : $this->mapContentElement($content)
        );
    }

    private function isGlobalContent(VehicleOfferContent $content): bool
    {
        return !empty($content->global_content_id);
    }

    private function mapGlobalContent(VehicleOfferContent $content): array
    {
        return [
            'ui_component' => $content->globalContent?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => [],

        ];
    }

    private function mapContentElement(VehicleOfferContent $content): array
    {
        return [
            'ui_component' => $content->contentElement?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => $content->subContent && $this->componentHasUiResource($content) === false
                ? $this->getContent($content->subContent)
                : [],
        ];
    }


    private function mapValue(VehicleOfferContent $content)
    {
        if ($this->componentHasUiResource($content)) {
            $contentResource = $this->getUiResource($content);
            return new $contentResource($content);
        }

        if (in_array($content->data['fieldId'] ?? '', ['image', 'images'])) {
            return collect($content->mediaUses)
                ->map(fn($mediaUse) => [
                    'src' => $mediaUse->getUrl('2560x650'), // TODO correct image size
                    'alt' => $mediaUse->alt_text,
                    'caption' => $mediaUse->caption,
                    'title' => $mediaUse->title,
                ]);
        }

        return $content->content;
    }

    private function componentHasUiResource(VehicleOfferContent $content): bool
    {
        if (!empty($content->global_content_id)) {
            return !empty($content->globalContent->ui_component)
                && class_exists(self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component);
        }

        return !empty($content->contentElement->ui_component)
            && class_exists(self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component);
    }

    private function getUiResource(VehicleOfferContent $content): string
    {
        return
            !empty($content->global_content_id)
                ? self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component
                : self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component;
    }

    private function componentFromField(VehicleOfferContent $content)
    {
        return $content->data['uiComponent'] ?? $this->primitiveField($content);
    }

    private function primitiveField(VehicleOfferContent $content)
    {
        return match ($content->data['fieldId'] ?? null) {
            'text-content' => 'UiArticleText',
            'text_area' => 'UiArticleText',
            'images' => 'UiArticleBanner',
            'image' => 'UiArticleBanner',
            default => null,
        };
    }

    private function fillSeo(): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $default = SeoDefault::query()->where('section', 'offer')->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->name,
                '{{MAKE}}' => $this->resource->make->name,
                '{{MODEL}}' => $this->resource->model->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{MAKE}}' => $this->resource->make->name,
                '{{MODEL}}' => $this->resource->model->name,
                '{{CONTENT_EXCERPT}}' => Str::limit($this->resource->description, 150),

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }

    private function getContactNumber(): ?array
    {
        if ($this->resource->dealership_id === null) {
            return null;
        }

        if ($this->resource->dealership->contact_no === null) {
            return null;
        }

        $countryCode = $this->resource->dealership->country;
//        $prefix = Cache::rememberForever(
//            'country-dial-code-' . $countryCode,
//            fn () => Country::query()->where('code', $countryCode)->first()?->dial_code
//        );

        $prefix = '+' . Country::query()->where('code', $countryCode)->first()?->dial_code;
        return [
            'number' => $prefix . $this->resource->dealership->contact_no,
            'display' => $prefix . ' ' . $this->displayNumber($this->resource->dealership->contact_no),
        ];
    }

    private function getDetails(): array
    {
        return [
            [
                'name' => __('labels.vehicle.colour'),
                'value' => $this->resource->colour
            ],
            [
                'name' => __('labels.vehicle.engine_size'),
                'value' => $this->resource->engine_size_cc ? $this->resource->engine_size_cc . 'cc' : null,
            ],
            [
                'name' => __('labels.vehicle.fuel_type'),
                'value' => $this->resource->fuelType?->name
            ],
            [
                'name' => __('labels.vehicle.body_style'),
                'value' => $this->resource->bodyStyle?->name
            ],
            [
                'name' => __('labels.vehicle.transmission'),
                'value' => $this->resource->transmission?->name
            ],
            [
                'name' => __('labels.vehicle.drivetrain'),
                'value' => $this->resource->drivetrain?->name
            ],
        ];
    }
}
