<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\HasPlaceholderImage;
use Carbon\Carbon;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleView;

class VehicleList extends JsonResource
{
    use HasPlaceholderImage;

    private Collection $placeholder;

    /**
     * Transform the resource collection into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    public function toArray($request)
    {
        $this->placeholder = $this->getPlaceholderImage('vehicle');

        $this->trackViews();
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource
                ->through(fn ($vehicle) => $this->mapVehicle($vehicle));
        } else {
            $this->resource = $this->resource
                ->map(fn($vehicle) => $this->mapVehicle($vehicle));
        }

        return parent::toArray($request);
    }

    public function mapVehicle(Vehicle $vehicle)
    {
        return [
            'id' => $vehicle->id,
            'title' => $vehicle->title,
            'derivative' => $vehicle->derivative,
            'slug' => $vehicle->slug,
            'is_new' => $vehicle->is_new,
            'price' => $vehicle->price,
            'monthly_price' => $vehicle->monthly_price,
            'make_id' => $vehicle->make_id,
            'thumbnail' => [
                'small' => $vehicle->getPreviewImage('325x250') ?? $this->placeholder->first()['src_medium'],
            ],
            'make' => Settings::get('automotive-vehicle-brand-on-filter-card')
                ? [
                    'name' => $vehicle->make->name ?? '',
                    'logo' => $vehicle->make->logo ?? '',
                ]
                : null,
            'location' => $this->locationValue($vehicle),
            'deposit' => $vehicle->deposit,
        ];
    }

    private function locationValue(Vehicle $vehicle): string
    {
        if (tenant()->hasMultipleLocations()) {
            return $vehicle->dealership->name ?? '';
        }

        return '';
    }


    private function trackViews(): void
    {
        /** @var Collection $ids */
        $ids = $this->resource->pluck('id');
        $existing = VehicleView::query()
            ->whereIn('vehicle_id', $ids)
            ->where('date', Carbon::now()->format('Y-m-d'))
            ->pluck('vehicle_id');

        $missing = $ids->reject(fn($id) => $existing->search($id));

        if ($missing->isNotEmpty()) {
            $data = $missing->map(fn ($id) => [
                'vehicle_id' => $id,
                'hits' => 0,
                'filter_views' => 1,
                'date' => Carbon::now()->format('Y-m-d'),
            ]);
            VehicleView::query()->upsert($data->toArray(), ['vehicle_id', 'date']);
        }

        VehicleView::query()
            ->whereIn('vehicle_id', $existing)
            ->increment('filter_views');
    }
}
