<?php

namespace Tests\Feature;

use App\Models\FilterIndex;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Models\FormQuestion;
use Mtc\Crm\Models\FormSection;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;

class OfferControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testListOffers()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create(['make_id' => $make->id]);
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $make2 = VehicleMake::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $make2->id,
            'filter_type' => 'make',
        ]);
        $model2 = VehicleModel::factory()->create(['make_id' => $make2->id]);
        FilterIndex::factory()->create([
            'filter_id' => $model2->id,
            'filter_type' => 'model',
        ]);

        VehicleOffer::factory(5)
            ->create([
                'make_id' => $make->id,
                'model_id' => $model->id,
                'published_at' => Carbon::now()->subDays(3),
            ]);
        VehicleOffer::factory(3)->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,

        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('offers.index', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('page', $response->json());
        $this->assertArrayHasKey('featured', $response->json());
        $this->assertArrayHasKey('makes', $response->json());
        $this->assertArrayHasKey('models', $response->json());
    }

    public function testSearchOffers()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        VehicleOffer::factory(5)
            ->create([
                'make_id' => $make->id,
                'model_id' => $model->id,
                'published_at' => Carbon::now()->subDays(3),
            ]);
        VehicleOffer::factory(3);
        $response = $this->asTenant(tenant())
            ->postJson(route('offers.search', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('make', $response->json());
        $this->assertEquals($make->name, $response->json('make'));
        $this->assertArrayHasKey('models', $response->json());
        $this->assertArrayHasKey('offers', $response->json());
    }

    public function testShowOffer()
    {
        $offer = VehicleOffer::factory()->create();
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
    }

    public function testShowOfferWithSeoData()
    {
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'seo' => ['title' => 'foo', 'description' => 'baz bar']
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('seo.title'));
        $this->assertEquals('baz bar', $response->json('seo.description'));

        SeoDefault::query()->create([
            'section' => 'offer',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);

        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertStringContainsString($offer->name, $response->json('seo.title'));
        $this->assertStringContainsString($make->name, $response->json('seo.title'));
    }

    public function testOfferHasForm()
    {
        $form = Form::factory()->create();
        $section = FormSection::factory()->create(['form_id' => $form->id]);
        FormQuestion::factory()->create([
            'form_id' => $form->id,
            'form_section_id' => $section->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);

        $offer = VehicleOffer::factory()->create();

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);

        $offer->update([
            'form_id' => $form->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('forms.enquire'));
        $this->assertEquals($form->id, $response->json('forms.enquire.id'));
    }

    public function testOfferShowsDealershipNumber()
    {
        $dealership1 = Dealership::factory()->create([
            'contact_no' => '123123',
        ]);
        $dealership2 = Dealership::factory()->create([
            'contact_no' => null
        ]);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create();
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $offer = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);
        $offer1 = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership1
        ]);
        $offer2 = VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'dealership_id' => $dealership2
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer1));
        $response->assertStatus(200);
        $this->assertIsArray($response->json('contact_number'));
        $this->assertArrayHasKey('number', $response->json('contact_number'));
        $this->assertArrayHasKey('display', $response->json('contact_number'));
        $this->assertStringContainsString('123123', $response->json('contact_number.number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('offers.show', $offer2));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));
    }
}
