<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleFinance;
use Mtc\MercuryDataModels\VehicleView;
use Tests\TestCase;

class VehicleControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFilterLoads()
    {
        Vehicle::factory(10)->create([
            'is_published' => true,
        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('sort_options', $response->json());
        $this->assertArrayHasKey('results', $response->json());
        $this->assertArrayHasKey('filters', $response->json());
        $this->assertEquals(10, $response->json('results.total'));
    }

    public function testViewSingleVehicle()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
    }

    public function testQuickViewVehicle()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.quick-view', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
        $this->assertArrayHasKey('actions', $response->json('data'));
        $this->assertArrayHasKey('odometer_mi', $response->json('data'));
        $this->assertArrayHasKey('odometer_km', $response->json('data'));
        $this->assertArrayHasKey('fuel_type', $response->json('data'));
        $this->assertArrayHasKey('transmission', $response->json('data'));

        $this->assertTrue(VehicleView::query()->where('vehicle_id', $vehicle->id)->where('quick_view_hits', 1)->exists());
    }

    public function testViewVehicleWithAccessKey()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => false,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));
        $response->assertStatus(403);
        $response = $this->asTenant(tenant())->getJson(
            route('vehicles.show', 'foo-baz-bar') . '?a=' . base64_encode($vehicle->id . '-' . $vehicle->slug)
        );
        $response->assertStatus(200);
    }

    public function testRecentlyViewed()
    {
        $vehicles = Vehicle::factory(3)->create(['is_published' => true]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.recentlyViewed', 'foo-baz-bar') . '?'
                . http_build_query(['slugs' => $vehicles->pluck('slug')->toArray()]));

        $response->assertStatus(200);
        $this->assertArrayHasKey('items', $response->json('recently_viewed'));
        $this->assertCount(3, $response->json('recently_viewed.items'));
    }

    public function testViewWithDefaultFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        VehicleFinance::factory()->create(['vehicle_id' => $vehicle->id]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertEquals('FinanceCalculator', $response->json('data.finance.value'));
        $this->assertArrayHasKey('change_values', $response->json('data.finance.data'));
        $this->assertArrayHasKey('values', $response->json('data.finance.data'));
    }

    public function testViewWithIvendiFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-ivendi-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('IVendiCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithCodeweaversFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-codeweavers-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('CodeweaversCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithEvolutionFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-evolution-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('EvolutionCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }
}
