<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\HasPlaceholderImage;
use Carbon\Carbon;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Events\LoadVehicleDataFields;
use Mtc\MercuryDataModels\Traits\HasVehicleCustomAttributes;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleView;

class VehicleList extends JsonResource
{
    use HasPlaceholderImage;
    use HasVehicleCustomAttributes;

    private Collection $placeholder;

    /**
     * Transform the resource collection into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    public function toArray($request)
    {
        $this->trackViews();
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource
                ->through(fn ($vehicle) => $this->mapVehicle($vehicle));
        } else {
            $this->resource = $this->resource
                ->map(fn($vehicle) => $this->mapVehicle($vehicle));
        }

        return parent::toArray($request);
    }

    public function mapVehicle(Vehicle $vehicle)
    {
        $data = [
            'id' => $vehicle->id,
            'title' => $vehicle->title,
            'derivative' => $vehicle->derivative,
            'slug' => $vehicle->slug,
            'is_new' => $vehicle->is_new,
            'price' => $vehicle->price,
            'make' => Settings::get('automotive-vehicle-brand-on-filter-card')
                ? [
                    'name' => $vehicle->make->name ?? '',
                    'logo' => $vehicle->make->logo ?? '',
                    'slug' => $vehicle->make->slug ?? '',
                ]
                : null,
            'location' => $this->locationValue($vehicle),
            'autoTraderData' => $vehicle->autoTraderData,
            'deposit' => $vehicle->deposit,
            'dealership' => $this->getDealership($vehicle),
            'fuel_type' => $vehicle->fuelType?->name,
            'transmission' => $vehicle->transmission?->name,
            'odometer_mi' => $vehicle->odometer_mi,
            'odometer_km' => $vehicle->odometer_km,
            'manufacture_year' => $vehicle->manufacture_year,
            'registration_number' => $vehicle->registration_number,
            'finance_example' => $this->getFinanceExample($vehicle),
            'model' => Settings::get('automotive-vehicle-model-on-filter-card')
                ? [
                    'name' => $vehicle->model->name ?? '',
                    'slug' => $vehicle->model->slug ?? '',
                ]
                : $vehicle->model?->name,
            'original_price' => $vehicle->original_price,
            'previous_price' => $vehicle->previous_price,
            'rrp_price' => $vehicle->rrp_price,
            'type' => $vehicle->type,
            'extra' => $this->getExtraData($vehicle),
            'seo' => $vehicle->seo,
            'age_identifier' => $vehicle->getAgeIdentifier(),
            'custom' => $vehicle->getAllCustom(),
            'monthly_price' => $vehicle->monthly_price,
            'monthly_cost_type' => $vehicle->getMonthlyCostType(),
        ];
        if (Settings::get('vehicle-card-image-count') > 1) {
            $data['gallery'] = $this->getGallery($vehicle, Settings::get('vehicle-card-image-count'));
        } else {
            $data['thumbnail'] = [
                'small' => $this->getPreviewImage($vehicle),
            ];
        }
        return $data;
    }

    private function getFinanceExample(Vehicle $vehicle)
    {
        if (Settings::get('vehicle-list-load-finance-example')) {
            return $vehicle->defaultFinanceExample;
        }

        return null;
    }

    private function getExtraData(Vehicle $vehicle)
    {
        return collect(event(new LoadVehicleDataFields($vehicle, 'ui')))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->merge($this->getCustomAttributes($vehicle));
    }

    private function getDealership(Vehicle $vehicle)
    {
        if (empty($vehicle->dealership)) {
            return null;
        }

        $dealership = $vehicle->dealership->only([
            'contact_no',
            'email',
            'name',
            'slug',
        ]);

        if (!empty($vehicle->dealership->franchise)) {
            $dealership['franchise'] = $vehicle->dealership->franchise->only([
                'logo',
                'name',
                'slug',
            ]);
        }

        return $dealership;
    }

    private function locationValue(Vehicle $vehicle): string
    {
        if (tenant()->hasMultipleLocations()) {
            return $vehicle->dealership->name ?? '';
        }

        return '';
    }

    private function trackViews(): void
    {
        /** @var Collection $ids */
        $ids = $this->resource->pluck('id');
        $existing = VehicleView::query()
            ->whereIn('vehicle_id', $ids)
            ->where('date', Carbon::now()->format('Y-m-d'))
            ->pluck('vehicle_id');

        $missing = $ids->reject(fn($id) => $existing->search($id));

        if ($missing->isNotEmpty()) {
            $data = $missing->map(fn ($id) => [
                'vehicle_id' => $id,
                'hits' => 0,
                'filter_views' => 1,
                'date' => Carbon::now()->format('Y-m-d'),
            ]);
            VehicleView::query()->upsert($data->toArray(), ['vehicle_id', 'date']);
        }

        VehicleView::query()
            ->whereIn('vehicle_id', $existing)
            ->increment('filter_views');
    }

    protected function largeImageDimensions(): string
    {
        return 'large';
    }

    protected function mediumImageDimensions(): string
    {
        return 'tile';
    }

    protected function thumbnailImageDimensions(): string
    {
        return 'tile';
    }
}
