<?php

namespace Tests\Feature;

use App\Models\FilterIndex;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\OfferType;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Template;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;

class FranchiseDataTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFranchiseData()
    {
        tenant()->update(['tier' => 'pro']);

        $make = VehicleMake::factory()->create();
        FilterIndex::factory()->create([
            'filter_type' => 'make',
            'filter_id' => $make->id,
            'slug' => 'foo'
        ]);
        $franchise = Franchise::factory()->create([
            'name' => 'foo',
            'slug' => 'baz',
            'make_id' => $make->id,
        ]);

        $page = Page::factory()->create([
            'status' => 'published',
            'franchise_id' => $franchise->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', $page->slug));

        $response->assertStatus(200);
        $this->assertTrue($response->json('isFranchise'));
        $this->assertEquals($make->id, $response->json('franchise.make_id'));
        $this->assertEquals('foo', $response->json('franchise.make_slug'));
    }

    public function testDealershipFranchise()
    {
        $franchise = Franchise::factory()->create([
            'name' => 'foo'
        ]);

        $dealership = Dealership::factory()->create([
            'franchise_id' => $franchise->id
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id
        ]);

        $this->assertEquals($franchise->name, $vehicle->dealership->franchise->name);
    }

    public function testGetFranchiseEnpoint()
    {
        tenant()->update(['tier' => 'pro']);
        $franchise = Franchise::factory()->create([
            'name' => 'foo',
            'slug' => 'baz'
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('franchises.show', $franchise->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('name', $response->json());
        $this->assertArrayHasKey('slug', $response->json());
        $this->assertArrayHasKey('logo', $response->json());
        $this->assertArrayHasKey('primary_colour', $response->json());
        $this->assertArrayHasKey('secondary_colour', $response->json());
        $this->assertArrayHasKey('menus', $response->json());
    }

    public function testGetFranchiseOffersEndpoint()
    {
        tenant()->update(['tier' => 'pro']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        $model = VehicleModel::factory()->create(['make_id' => $make->id]);
        $index = FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);
        $franchise = Franchise::factory()->create([
            'name' => 'foo',
            'slug' => 'baz',
            'make_id' => $make->id,
        ]);

        $type = OfferType::factory()->create();
        VehicleOffer::factory(5)->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'featured' => 0,
            'type_id' => $type->id,
        ]);
        VehicleOffer::factory(5)->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'featured' => 1,
            'type_id' => $type->id,
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('franchises.offers', $franchise->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('featured', $response->json());
        $this->assertArrayHasKey('models', $response->json());
        $this->assertArrayHasKey($model->filterIndex->slug, $response->json('models'));
        $this->assertCount(5, $response->json('featured'));
        $this->assertArrayHasKey('slug', $response->json('featured.0'));
        $this->assertArrayHasKey('name', $response->json('featured.0'));
    }
}
