<?php

namespace App\Http\Controllers;

use App\Filter;
use App\Http\Requests\OfferSearchRequest;
use App\Http\Requests\ViewOfferRequest;
use App\Http\Resources\MakeListResource;
use App\Http\Resources\OfferListResource;
use App\Http\Resources\OfferResource;
use App\Http\Resources\PageResource;
use App\Traits\ModelsForMakeInSearch;
use Carbon\Carbon;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;

class OfferController extends Controller
{
    use ModelsForMakeInSearch;

    public function index(OfferSearchRequest $request): array
    {
        $page = Page::query()->where('slug', 'offers')->first();
        $featured = VehicleOffer::query()->active()
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
                'fuelType',
                'transmission',
                'drivetrain',
                'bodyStyle',
            ])
            ->where('featured', 1)
            ->get();

        $makes = VehicleMake::query()
            ->with('filterIndex')
            ->whereHas('offers', fn ($query) => $query->active())
            ->get();

        $models = VehicleModel::query()
            ->whereHas('cheapestActiveOffer')
            ->with([
                'cheapestActiveOffer.primaryMediaUse.media',
                'cheapestActiveOffer.make.filterIndex',
                'cheapestActiveOffer.model.filterIndex',
                'make.filterIndex',
                'cheapestActiveOffer.make',
                'cheapestActiveOffer.model',
                'cheapestActiveOffer.fuelType',
                'cheapestActiveOffer.transmission',
                'cheapestActiveOffer.drivetrain',
                'cheapestActiveOffer.bodyStyle',
            ])
            ->get()
            ->groupBy(fn ($model) => $model->make->filterIndex->slug)
            ->map(fn ($group) => new OfferListResource($group->map(fn ($model) => $model->cheapestActiveOffer)));

        return [
            'page' => new PageResource($page),
            'featured' => new OfferListResource($featured),
            'makes' => new MakeListResource($makes),
            'models' => $models,
        ];
    }

    public function search(OfferSearchRequest $request, string $make): array
    {
        $makeModel = Filter::matchSlugToRecord('make', $make);
        $vehicleModel = $request->filled('model_id')
            ? Filter::matchSlugToRecord('model', $request->input('model_id'))
            : null;
        $makeId = $makeModel->id;
        $offers = VehicleOffer::query()
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
            ])
            ->where('make_id', $makeId)
            ->whereModelSlug($request->input('model_id'))
            ->setSortBy($request->input('sort_by'))
            ->active()
            ->paginate();

        return [
            'make' => $makeModel->name,
            'model' => $vehicleModel?->name,
            'models' => $this->getModelsForMake($makeId, 'offers'),
            'offers' => new OfferListResource($offers),
        ];
    }

    public function show(ViewOfferRequest $request, VehicleOffer $offer): OfferResource
    {
        $view = $offer->views()
            ->firstOrNew(['date' => Carbon::now()->format('Y-m-d')]);
        $view->hits++;
        $view->save();
        return new OfferResource($offer);
    }
}
