<?php

namespace App\Http\Resources;

use App\Traits\HasPlaceholderImage;
use App\Traits\RetrievesContent;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Country;

class NewCarResource extends JsonResource
{
    use HasPlaceholderImage;
    use FranchiseData;
    use PhoneNumberDisplay;
    use RetrievesContent;

    public static $wrap = '';

    private const UI_RESOURCE_PREFIX = 'App\\Http\\Resources\\';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->load([
            'mediaUses.media',
            'transmission',
            'fuelType',
            'bodyStyle',
            'drivetrain',
            'form.sections.questions',
            'content.mediaUses.media',
            'content.subContent.mediaUses.media',
            'content.subContent.subContent.mediaUses.media',
            'content.subContent.subContent.subContent.mediaUses.media',
            'content.contentElement',
            'content.globalContent.contentElement',
            'content.subContent.contentElement',
            'content.subContent.globalContent.contentElement',
            'content.subContent.subContent.contentElement',
            'content.subContent.subContent.globalContent.contentElement',
            'content.subContent.subContent.subContent.contentElement',
            'content.subContent.subContent.subContent.globalContent.contentElement',
            'content.subContent.subContent.subContent.subContent.contentElement',
            'content.subContent.subContent.subContent.subContent.globalContent.contentElement',
        ]);
        $this->resource->make_name = $this->resource->make?->name;
        $this->resource->model_name = $this->resource->model?->name;
        $this->resource->view_stock = false;

        $this->resource->forms = [
            'enquire' => $this->resource->form_id
                ? (new FormViewResource($this->resource->form))->setValues(['new_car_id' => $this->resource->id])
                : null,
        ];

        $resource = $this->resource->only([
            'name',
            'price',
            'make_name',
            'model_name',
            'derivative',
            'content',
            'media_uses',
            'forms',
            'finance',
            'specs',
            'features',
            'hero',
        ]);

        $resource['media_uses'] = $this->getGallery($this->resource);
        $resource['details'] = $this->getDetails();

        $resource['contact_number'] = $this->getContactNumber();
        $resource['content'] = $this->getContent($this->resource->content);
        $resource['seo'] = $this->fillSeo('new-car');
        $resource['isFranchise'] = $this->isFranchise();
        $resource['franchise'] = $this->franchiseData();
        $resource['pageContent'] = [
            'title' => __('automotive.new_cars_for_make', ['make' => $this->resource->make_name]),
            'description' => '',
        ];
        return $resource;
    }

    protected function getGallery(Model $model): Collection
    {
        if (is_null($model->mediaUses) || $model->mediaUses->isEmpty()) {
            return $this->getPlaceholderForGallery($model);
        }

        return $this->resource->mediaUses
            ->filter(fn ($mediaUse) => $mediaUse->flags['hero'] ?? null)
            ->map(fn($mediaUse) => [
                'title' => $mediaUse->title,
                'alt_text' => $mediaUse->alt_text,
                'description' => $mediaUse->description,
                'caption' => $mediaUse->caption,
                'src' => $mediaUse->getUrl($this->largeImageDimensions()),
                'src_large' => $mediaUse->getUrl($this->mediumImageDimensions()),
                'type' => $mediaUse->media->type
            ]);
    }

    private function getDetails(): array
    {
        return [
            [
                'name' => __('labels.vehicle.colour'),
                'value' => $this->resource->colour
            ],
            [
                'name' => __('labels.vehicle.engine_size'),
                'value' => $this->resource->engine_size_cc ? $this->resource->engine_size_cc . 'cc' : null,
            ],
            [
                'name' => __('labels.vehicle.fuel_type'),
                'value' => $this->resource->fuelType?->name
            ],
            [
                'name' => __('labels.vehicle.body_style'),
                'value' => $this->resource->bodyStyle?->name
            ],
            [
                'name' => __('labels.vehicle.transmission'),
                'value' => $this->resource->transmission?->name
            ],
            [
                'name' => __('labels.vehicle.drivetrain'),
                'value' => $this->resource->drivetrain?->name
            ],
        ];
    }

    protected function largeImageDimensions(): string
    {
        return 'full-width-banner';
    }

    protected function mediumImageDimensions(): string
    {
        return 'full-width-banner';
    }
}
