<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\IntegrationRepository;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\GlobalContent;
use Mtc\MercuryDataModels\Menu;
use Mtc\MercuryDataModels\SeoRedirect;

class SiteResource extends JsonResource
{
    use FranchiseData;
    use PhoneNumberDisplay;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $integrations = new IntegrationRepository();

        $menus = Menu::query()
            ->with('entries.children.children.children')
            ->get()
            ->keyBy('slug');

        return [
            'id' => tenant('id'),
            'name' => Settings::get('app-name'),
            'locale' => 'en-GB',
            'logo' => Settings::get('site-alternate-logo'),
            'contact_phone_number' => $this->displayNumber(Settings::get('app-contact-phone_number') ?? ''),
            'currency' => Settings::get('app-details-currency'),
            'favicon' => Settings::get('site-favicon'),
            'apple-touch' => Settings::get('site-apple-touch-icon'),
            'suspended' => tenant('suspended_at'),
            'branding' => [
                'primaryColour' => '#' . ltrim(Settings::get('app-primary-colour'), '#'),
                'secondaryColour' => '#' . ltrim(Settings::get('app-secondary-colour'), '#'),
                'tertiaryColour' => '#' . ltrim(Settings::get('app-tertiary-colour'), '#'),
                'successColour' => '#' . ltrim(Settings::get('app-success-colour'), '#'),
                'errorColour' => '#' . ltrim(Settings::get('app-error-colour'), '#'),
                'informationColour' => '#' . ltrim(Settings::get('app-information-colour'), '#'),
                'textColour' => '#' . ltrim(Settings::get('app-text-colour'), '#'),
                'subTextColour' => '#' . ltrim(Settings::get('app-subtext-colour'), '#'),
                'layerDarkColour' => '#' . ltrim(Settings::get('app-layer-dark-colour'), '#'),
                'layerAccentColour' => '#' . ltrim(Settings::get('app-layer-accent-colour'), '#'),
                'layerLightColour' => '#' . ltrim(Settings::get('app-layer-light-colour'), '#'),
            ],
            'restricted_access' => Settings::get('site-access-enabled'),
            'scripts' => $this->getScripts(),
            'redirects' => SeoRedirect::query()
                ->select(['from', 'to', 'code'])
                ->get(),
            'header' => [
                'isFranchise' => $this->isFranchise(),!
                'franchise' => $this->franchiseData(),
                'siteName' => Settings::get('app-name'),
                'logo' => Settings::get('site-logo'),
                'mainMenu' => new MenuResource($menus['header-main'] ?? []),
                'sidebarMenu' => new MenuResource($menus['header-sidebar'] ?? []),
                'franchiseMenu' => new MenuResource($menus['franchise-menu'] ?? []),
            ],
            'cookie_content' => $this->loadGlobal('cookie-consent'),
            'footer' => [
                'logo' => Settings::get('site-logo'),
                'menu' => new MenuResource($menus['footer'] ?? []),
                'copyright' => $this->loadGlobal('copyright')->first(),
                'footer_data' => $this->loadGlobal('footer'),
                'socials' => $integrations->findNonEmptyValuesForType('socials')
                    ->map(fn ($link, $key) => [
                        'key' => $key,
                        'url' => $link,
                    ])->values(),
                'subscribe' => [
                    'enabled' => $integrations->getEnabledForType('newsletters')->isNotEmpty(),
                ],

            ],
            'liveChat' => $this->liveChat(),
        ];
    }

    private function liveChat(): ?LiveChatResource
    {
        if ($this->hasLiveChatEnabled() === false) {
            return null;
        }

        return $this->getEnabledLiveChatWidget();
    }

    private function hasLiveChatEnabled(): bool
    {
        return (new IntegrationRepository())->getEnabledForType('chat')->isNotEmpty();
    }

    private function getEnabledLiveChatWidget(): LiveChatResource
    {
        return new LiveChatResource((new IntegrationRepository())->getEnabledForType('chat')->first());
    }

    private function getScripts(): array
    {
        $scripts =  (new IntegrationRepository())->getEnabledForType('tracking')
            ->filter(fn($integration) => $integration['class'])
            ->map(fn($integration) => App::make($integration['class']))
            ->values()
            ->toArray();

        return collect($scripts)
            ->flatten(1)
            ->toArray();
    }

    private function loadGlobal(string $slug)
    {
        return collect(GlobalContent::query()
            ->where('slug', $slug)
            ->first()
            ?->content ?? [])
            ->keyBy(fn($entry) => $entry['slug'])
            ->map(fn($entry) => $entry['content']);
    }
}
