<?php

namespace App\Http\Resources\VehicleAction;

use App\Facades\Settings;
use App\IntegrationRepository;
use App\Sales\SilverBullet;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\Vehicle;

class FinanceFactory
{
    public function get(Vehicle $vehicle)
    {
        if ($this->hasKeyloopFinance()) {
            return App::make(SilverBullet::class)->getVehicleAction($vehicle, 'finance');
        }
        return false;
    }

    public function widget(Vehicle $vehicle)
    {
        return match ($this->enabledFinanceProvider()) {
            'keyloop' => $this->keyloopWidget(),
            'ivendi' => $this->ivendiWidget($vehicle),
            'code-weavers' => $this->codeWeaversWidget($vehicle),
            'evolution-funding' => $this->evolutionWidget($vehicle),
            default => $this->defaultCalculator($vehicle),
        };
    }

    private function hasKeyloopFinance(): bool
    {
        return Settings::get('sales-silver-bullet-enabled')
            && Settings::get('sales-silver-bullet-finance');
    }

    private function keyloopWidget()
    {
        return null;
    }

    private function ivendiWidget(Vehicle $vehicle)
    {
        return [
            'type' => 'component',
            'value' => 'IVendiCalculator',
            'data' => [
                'username' => Settings::get('finance-ivendi-api_key'),
                'quotee_uid' => $vehicle->dealership?->data['ivendi'] ?? null,
                'class' => $vehicle->type,
                'is_new' => $vehicle->is_new,
                'registration_number' => $vehicle->registration_number,
                'first_registration_date' => $vehicle->first_registration_date,
                'cap_code' => $vehicle->cap_code,
                'cap_id' => $vehicle->cap_id,
                'price' => $vehicle->price,
                'vat_included' => $vehicle->type !== 'lcv',
                'vat_qualifying' => true,
                'mileage' => $vehicle->odometer_mi,
                'image' => $vehicle->getPreviewImage('large'),
                'deposit' => Settings::get('finance-deposit_percentage'),
                'term' => Settings::get('finance-term'),
                'annual_mileage' => Settings::get('finance-annual_mileage'),
            ],
        ];
    }

    private function codeWeaversWidget(Vehicle $vehicle)
    {
        return [
            'type' => 'component',
            'value' => 'CodeweaversCalculator',
            'data' => [
                'key' => Settings::get('finance-codeweavers-api_key'),
                'type' => $vehicle->type,
                'is_new' => $vehicle->is_new,
                'registration_number' => $vehicle->registration_number,
                'first_registration_date' => $vehicle->first_registration_date,
                'cap_id' => $vehicle->cap_id,
                'price' => $vehicle->price,
                'mileage' => $vehicle->odometer_mi,
                'image' => $vehicle->getPreviewImage('large'),
                'deposit' => Settings::get('finance-deposit_percentage'),
                'term' => Settings::get('finance-term'),
                'annual_mileage' => Settings::get('finance-annual_mileage'),

            ],
        ];
    }

    private function evolutionWidget(Vehicle $vehicle)
    {
        return [
            'type' => 'component',
            'value' => 'EvolutionCalculator',
            'data' => [
                'key' => Settings::get('finance-evolution-client_id'),
                'type' => $vehicle->type,
                'is_new' => $vehicle->is_new,
                'make' => $vehicle->make->name,
                'model' => $vehicle->model->name,
                'derivative' => $vehicle->derivative,
                'registration_number' => $vehicle->registration_number,
                'first_registration_date' => $vehicle->first_registration_date,
                'vat_applicable' => $vehicle->type !== 'lcv',
                'cap_id' => $vehicle->cap_id,
                'price' => $vehicle->price,
                'mileage' => $vehicle->odometer_mi,
                'image' => $vehicle->getPreviewImage('large'),
                'deposit' => Settings::get('finance-deposit_percentage'),
                'term' => Settings::get('finance-term'),
                'annual_mileage' => Settings::get('finance-annual_mileage'),
            ],
        ];
    }

    private function enabledFinanceProvider()
    {
        return (new IntegrationRepository())->getEnabledForType('finance')
            ->keys()
            ->first();
    }

    private function defaultCalculator(Vehicle $vehicle)
    {
        return [
            'type' => 'component',
            'value' => 'FinanceCalculator',
            'data' => [
                'change_values' => false,
                'values' => $this->mapValues($vehicle->financeExamples()
                    ->orderBy('monthly_price')
                    ->first()),
            ]
        ];
    }

    private function mapValues(?Model $example)
    {
        if (!$example) {
            return null;
        }

        return collect($example->getFillable())
            ->reject(fn($field) => in_array($field, ['vehicle_id', 'provider']))
            ->map(fn($field) => [
                'id' => $field,
                'is_price' => in_array($field, ['deposit', 'first_payment', 'final_payment', 'monthly_price']),
                'name' => __('labels.finance.' . $field),
                'value' => $this->formatValue($field, $example[$field]),
            ])
            ->toArray();
    }

    private function formatValue($field, mixed $value)
    {
        if (in_array($field, ['finance_type', 'apply_url'])) {
            return $value;
        }

        if (in_array($field, ['apr', 'interest_rate'])) {
            return round($value, 1) . '%';
        }

        return round($value, 0);
    }
}
