<?php

namespace App\Http\Resources;

use App\Http\Resources\VehicleAction\FinanceFactory;
use App\Http\Resources\VehicleAction\LoadsVehicleActions;
use App\Traits\HasPlaceholderImage;
use App\Traits\RetrievesContent;
use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Events\LoadVehicleDataFields;
use Mtc\MercuryDataModels\Traits\HasVehicleCustomAttributes;

class VehicleResource extends JsonResource
{
    use LoadsVehicleActions;
    use HasPlaceholderImage;
    use RetrievesContent;
    use HasVehicleCustomAttributes;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->fuel_type = $this->resource->fuelType?->name;
        $this->resource->transmission = $this->resource->transmission?->name;
        $this->resource->body_style = $this->resource->bodyStyle?->name;
        $this->resource->load([
            'mediaUses.media',
            'specs',
            'features',
            'equipment'
        ]);
        $this->resource->specs = $this->resource->specs->map(fn($spec) => [
            'category' => $spec->category,
            'description' => $spec->description,
            'value' => $spec->value,
        ]);

        $this->resource->features = $this->resource->features->pluck('name');
        $this->resource->actions = $this->getVehicleActions($this->resource);

        if ($this->resource->dealership) {
            $this->resource->dealership->address = collect([
                $this->resource->dealership->address1,
                $this->resource->dealership->address2,
                $this->resource->dealership->city,
                $this->resource->dealership->postcode,
            ])->filter()->implode(', ');
        }

        $this->resource->dealership = $this->resource->dealership?->only([
            'slug',
            'name',
            'contact_no',
            'address',
            'is_open',
            'opens_at',
            'closes_at',
        ]);

        $this->resource->content = $this->getContent();

        $resource = $this->resource->only([
            'actions',
            'autoTraderData',
            'battery_capacity_kwh',
            'battery_charge_time',
            'battery_quick_charge_time',
            'battery_range',
            'battery_usable_capacity_kwh',
            'bhp',
            'bodyStyle',
            'co2',
            'colour',
            'content',
            'dealership',
            'deposit',
            'derivative',
            'door_count',
            'drivetrain',
            'engine_size_cc',
            'equipment',
            'features',
            'financeExamples',
            'fuel_type',
            'gross_vehicle_weight_kg',
            'main_video_url',
            'make',
            'manufacture_year',
            'model',
            'mpg',
            'odometer_mi',
            'odometer_km',
            'payload_kg',
            'media_uses',
            'monthly_price',
            'original_price',
            'price',
            'registration_number',
            'rrp_price',
            'seats',
            'specs',
            'title',
            'transmission',
            'type',
            'wheelbase_mm',
            'wheelbase_type',

        ]);

        $resource['media_uses'] = $this->getGallery($this->resource, 'large');
        $resource['monthly_cost_type'] = $this->getMonthlyCostType();
        $resource['finance'] = $this->getFinanceData();
        $resource['extra'] = $this->getExtraData();
        $resource['seo'] = $this->fillSeo('vehicle');
        $resource['age_identifier'] = $this->getAgeIdentifier();
        $resource['lez_compliant'] = $this->resource->getCustom('lez_compliant');

        return $resource;
    }

    private function getExtraData()
    {
        return collect(event(new LoadVehicleDataFields($this->resource, 'ui')))
            ->filter()
            ->mapWithKeys(fn($field) => $field)
            ->merge($this->getCustomAttributes());
    }

    private function getContent(): Collection
    {
        return collect([
            [
                'title' => __('labels.description'),
                'content' => $this->resource->description,
                'buttonLink' => false
            ],
        ])->filter(fn($entry) => !empty($entry['content']));
    }

    private function getMonthlyCostType(): ?string
    {
        if (empty($this->resource->monthly_cost_type)) {
            return null;
        }

        return collect(config('automotive.monthly_cost_types', []))
            ->filter(fn($type) => $type['id'] === $this->resource->monthly_cost_type)
            ->first()['name'] ?? null;
    }

    private function getFinanceData()
    {
        return (new FinanceFactory())->widget($this->resource);
    }

    protected function largeImageDimensions(): string
    {
        return 'large';
    }

    protected function mediumImageDimensions(): string
    {
        return 'medium';
    }

    /**
     * Get the UK age identifier from the registration date.
     * Age identifier should be a 2 digit number.
     *
     * The year runs from March to March for registrations.
     *
     * If reg date between start of March and end of August,
     * age identifier is reg year in 2 digit format.
     *
     * If reg date between start of September and end of December,
     * age identifier is (reg year + 50) in 2 digit format.
     *
     * If reg date between start of January and end of March,
     * age identifier is (previous reg year + 50) in 2 digit format,
     * as the registration belongs to the previous year.
     *
     * @return int|string
     */
    private function getAgeIdentifier()
    {
        if (
            empty($this->resource->first_registration_date)
            || $this->resource->first_registration_date->format('Y') < 2001
        ) {
            return '';
        }

        $reg_year = $this->resource->first_registration_date->format('y');
        $reg_month = $this->resource->first_registration_date->format('m');

        if ($reg_month < 3) {
            // the identifier should relate to the previous year,
            // as the year for registrations runs March to March
            return ($reg_year - 1) + 50;
        }

        if ($reg_month >= 3 && $reg_month < 9) {
            return $reg_year;
        }

        if ($reg_month >= 9) {
            return $reg_year + 50;
        }

        return '';
    }
}
