<?php

namespace App\Modules\PlaceholderImages;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class ImaginStudio
{
    public function getForVehicle(Vehicle $vehicle, ?int $angle = 1): string
    {
        $colour = $this->determineColour($vehicle->make?->name, $vehicle->colour);
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
                'customer' => config('services.imagin-studio.key'),
                'make' => $vehicle->make?->name,
                'modelFamily' => $vehicle->model?->name,
                'modelVariant' => $vehicle->derivative,
                'modelYear' => $vehicle->manufacture_year,
                'powerTrain' => $vehicle->fuelType?->name,
                'transmission' => $vehicle->transmissionType?->name,
                'bodySize' => $vehicle->bodyStyle?->name,
                'angle' => $angle,
                'paintId' => $colour,
                'aspectRatio' => '16:9',
            ]);
    }

    public function getForOffer(VehicleOffer $offer, ?int $angle = 1): string
    {
        $colour = $this->determineColour($offer->make?->name, $offer->colour);
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
                'customer' => config('services.imagin-studio.key'),
                'make' => $offer->make?->name,
                'modelFamily' => $offer->model?->name,
                'powerTrain' => $offer->fuelType?->name,
                'transmission' => $offer->transmissionType?->name,
                'bodySize' => $offer->bodyStyle?->name,
                'paintId' => $colour,
                'angle' => $angle,
                'aspectRatio' => '3:2',
            ]);
    }

    public function getForNewCar(NewCar $newCar, ?int $angle = 1): string
    {
        $colour = $this->determineColour($newCar->make?->name, $newCar->colour);
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
                'customer' => config('services.imagin-studio.key'),
                'make' => $newCar->make?->name,
                'modelFamily' => $newCar->model?->name,
                'powerTrain' => $newCar->fuelType?->name,
                'transmission' => $newCar->transmissionType?->name,
                'bodySize' => $newCar->bodyStyle?->name,
                'paintId' => $colour,
                'angle' => $angle,
                'aspectRatio' => '3:2',
            ]);
    }

    public function getForOther(string $make, string $model, ?string $colour): string
    {
        if ($colour) {
            $colour = $this->determineColour($make, $colour);
        }
        return 'https://cdn.imagin.studio/getImage?' . http_build_query([
                'customer' => config('services.imagin-studio.key'),
                'make' => $make,
                'modelFamily' => $model,
                'paintId' => $colour,
                'aspectRatio' => '16:10',
            ]);
    }

    private function determineColour(?string $make, ?string $colour): ?string
    {
        if ($colour === null) {
            return null;
        }

        $colour = strtolower(Str::slug($colour));
        $manufacturerColours = $this->getManufacturerColours($make);
        return collect($manufacturerColours['paintData']['paintCombinations'] ?? [])
            ->filter(function ($paintSwatch) use ($colour) {
                return ($paintSwatch['primary']['colourCluster'] ?? null) === $colour
                    || collect($paintSwatch['mapped'] ?? [])
                        ->filter(fn($modelColours) => in_array($colour, $modelColours['nativePaintDescriptions']))
                        ->isNotEmpty();
            })
            ->keys()
            ->first();
    }

    private function getManufacturerColours(?string $make): Collection
    {
        return Cache::remember('imagin-colours-for-' . Str::slug($make), now()->addDay(), function () use ($make) {
            $url = 'https://cdn.imagin.studio/getPaints?' . http_build_query([
                    'customer' => config('services.imagin-studio.key'),
                    'target' => 'make',
                    'make' => $make
                ]);
            return collect(json_decode(file_get_contents($url), true));
        });
    }
}
