<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\MercuryDataModels\GlobalContent;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Seo404;
use Tests\TestCase;

class PageControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testShowPageEndpoint()
    {
        Page::factory()->create(['title' => 'foo baz bar', 'slug' => 'home', 'status' => 'published']);

        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', 'home'));
        $response->assertStatus(200);

        $this->assertEquals('home', $response->json('slug'));
        $this->assertEquals('foo baz bar', $response->json('title'));
    }

    public function testShowGlobalContentEndpoint()
    {
        GlobalContent::factory()->create([
            'name' => 'foo baz bar',
            'slug' => 'foo',
            'content' => [
                'foo' => [
                    'id' => 1,
                    'content' => 'baz'
                ]
            ]
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('pages.get-global', 'faz'));
        $response->assertStatus(404);

        $response = $this->asTenant(tenant())
            ->getJson(route('pages.get-global', 'foo'));
        $response->assertStatus(200);

        $this->assertEquals('baz', $response->json('foo.content'));
    }

    public function testGetSiteEndpoint()
    {
        Page::factory()->create(['title' => 'foo baz bar', 'slug' => 'home']);

        $response = $this->asTenant(tenant())
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertEquals(tenant('id'), $response->json('id'));
    }
    public function testPagePreviewRestriction()
    {
        $page = Page::factory()->create([
            'title' => 'foo baz bar',
            'slug' => 'home',
            'status' => 'published'
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', 'home'));
        $response->assertStatus(200);
        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', 'home') . '?a=1231231231');
        $response->assertStatus(200);

        $page->update(['status' => 'draft']);
        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', 'home'));
        $response->assertStatus(401);

        $access = base64_encode($page->id . '-' . $page->slug);
        $response = $this->asTenant(tenant())
            ->getJson(route('pages.show', 'home') . '?a=' . $access);
        $response->assertStatus(200);
    }

    public function test404Tracking()
    {
        $this->assertEquals(0, Seo404::query()->where('path', '/foo-faz')->count());
        $response = $this->asTenant(tenant())
            ->withHeaders(['x-path' => '/foo-faz'])
            ->getJson(route('pages.track-404'));

        $response->assertStatus(200);

        $this->assertEquals(1, Seo404::query()->where('path', '/foo-faz')->count());
        $this->assertEquals(1, Seo404::query()->where('path', '/foo-faz')->pluck('hits')->first());

        $response = $this->asTenant(tenant())
            ->withHeaders(['x-path' => '/foo-faz'])
            ->getJson(route('pages.track-404'));
        $this->assertEquals(1, Seo404::query()->where('path', '/foo-faz')->count());
        $this->assertEquals(2, Seo404::query()->where('path', '/foo-faz')->pluck('hits')->first());
    }
}
