<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;
use App\Facades\Settings;

class IntegrationTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testGetIntegrationToken()
    {
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-enabled')
            ->andReturn(true);
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-client_id')
            ->andReturn('client-id');
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-client_secret')
            ->andReturn('client-secret');

        Http::fake([
            'https://br-api.aos.tv/token/' => Http::response([
                'access_token' => 'mocked-token',
                'expires_in' => 3600,
            ], 200),
        ]);

        Cache::forget('autos_on_show_token_' . tenant('id'));

        $response = $this->asTenant(tenant())
            ->postJson(route('integrations.get-token'), [
                'integration' => 'autos-on-show',
            ]);

        $response->assertOk();
        $response->assertJson([
            'access_token' => 'mocked-token',
        ]);
    }

    /** @test */
    public function it_fails_if_integration_is_disabled()
    {
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-enabled')
            ->andReturn(false);

        $response = $this->asTenant(tenant())
            ->postJson(route('integrations.get-token'), [
                'integration' => 'autos-on-show',
            ]);

        $response->assertStatus(400);
        $response->assertJson([
            'error' => 'Integration does not support token or is disabled.',
        ]);
    }

    /** @test */
    public function it_returns_404_for_unknown_integration()
    {
        $response = $this->asTenant(tenant())
            ->postJson(route('integrations.get-token'), [
                'integration' => 'not-existing',
            ]);

        $response->assertStatus(400);
        $response->assertJson([
            'error' => 'Integration does not support token or is disabled.',
        ]);
    }

    /** @test */
    public function it_handles_http_failure_from_token_endpoint()
    {
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-enabled')
            ->andReturn(true);
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-client_id')
            ->andReturn('client-id');
        Settings::shouldReceive('get')
            ->with('image-sync-autos-on-show-client_secret')
            ->andReturn('client-secret');

        Http::fake([
            'https://br-api.aos.tv/token/' => Http::response('Unauthorized', 401),
        ]);

        Cache::forget('autos_on_show_token_' . tenant('id'));

        $response = $this->asTenant(tenant())
            ->postJson(route('integrations.get-token'), [
                'integration' => 'autos-on-show',
            ]);

        $response->assertStatus(500);
        $response->assertJsonFragment(['error' => 'Failed to fetch token: 401 - Unauthorized']);
    }
}
