<?php

namespace App\Http\Controllers;

use App\Facades\Settings;
use Mtc\Filter\Facades\Filter;
use App\Http\Requests\ViewVehicleRequest;
use App\Http\Resources\VehicleQuickViewResource;
use App\Http\Resources\VehicleResource;
use App\Traits\CacheObject;
use App\VehicleRepository;
use Mtc\MercuryDataModels\Vehicle;
use App\Repositories\RecentlyViewed;
use Illuminate\Http\Request;

class VehicleController extends Controller
{
    use CacheObject;

    /**
     * Show vehicle FPA details
     *
     * @param Vehicle $vehicle
     * @return array
     */
    public function show(
        ViewVehicleRequest $request,
        Vehicle $vehicle,
        RecentlyViewed $recentlyViewed,
        VehicleRepository $repository
    ) {
        $repository->trackView($vehicle);
        $recentlyViewed->add('vehicles', $vehicle->id);
        return $this->cache('vehicle-show-' . $vehicle->id, 2, function () use ($vehicle) {
            $crossSell = [];
            if (
                Settings::get('vehicles-cross-sell-enabled')
                && in_array(Settings::get('vehicles-cross-sell-location', ''), ['all', 'fpa'])
            ) {
                $crossSell = [
                    'name' => __('automotive.cross-sell.similar'),
                    'items' => (new VehicleRepository())->similar(
                        $vehicle,
                        4,
                        Settings::get('vehicles-cross-sell-type')
                    ),
                ];
            }
            return [
                'data' => new VehicleResource($vehicle),
                'cross_sell' => $crossSell,
            ];
        });
    }

    public function quickView(
        ViewVehicleRequest $request,
        Vehicle $vehicle,
        VehicleRepository $repository
    ) {
        $repository->trackQuickView($vehicle);
        return [
            'data' => new VehicleQuickViewResource($vehicle),
        ];
    }

    /**
     * Get list of vehicles for compare
     *
     * @param Request $request
     * @param VehicleRepository $repository
     * @return array
     */
    public function compare(Request $request, VehicleRepository $repository): array
    {
        $vehicleSlugs = $request->input('slugs') ?? [];

        return [
            'data' => $repository->compare($vehicleSlugs),
        ];
    }

    /**
     * Get list of recently viewed vehicles
     *
     * @param string $vehicle
     * @param Request $request
     * @return array
     */
    public function recentlyViewed(string $vehicle, Request $request): array
    {
        $vehicleModel = Vehicle::query()->where('slug', $vehicle)->first();
        $vehicleSlugs = $request->input('slugs') ?? [];
        $limit = $request->input('limit', 4);

        return [
            'recently_viewed' => [
                'name' => __('automotive.recently_viewed'),
                'items' => (new VehicleRepository())->recentlyViewed($vehicleModel, $vehicleSlugs, $limit),
            ]
        ];
    }

    /**
     * Get list of recently added vehicles
     *
     * @param Request $request
     * @return array
     */
    public function recentlyAdded(Request $request): array
    {
        $limit = $request->input('limit') ?? 4;
        $is_new = $request->input('is_new');

        $results = (new VehicleRepository())->recentlyAdded(
            $limit,
            array_merge($request->input(), [
                'is_new' => $is_new
            ])
        );

        return [
            'recently_added' => [
                'name' => __('automotive.recently_added'),
                'items' => $results,
            ],
            'finance_example' => Filter::getRepresentativeFinance($results)
        ];
    }

    /**
     * Get list of featured vehicles
     *
     * @param Request $request
     * @param Vehicle $vehicle
     * @param VehicleRepository $repository
     * @return array
     */
    public function featured(Request $request, Vehicle $vehicle, VehicleRepository $repository): array
    {
        return [
            'name' => __('automotive.cross-sell.featured'),
            'items' => $repository
                ->featured(
                    !empty($vehicle->id) ? $vehicle : null,
                    $request->input('limit') ?? 4,
                    $request->input()
                )
                ->toArray($request),
        ];
    }

    /**
     * Get list of similar vehicles
     *
     * @param Request $request
     * @param Vehicle $vehicle
     * @param VehicleRepository $repository
     * @return array
     */
    public function similar(Request $request, Vehicle $vehicle, VehicleRepository $repository): array
    {
        return [
            'name' => __('automotive.cross-sell.similar'),
            'items' => $repository
                ->similar(
                    $vehicle,
                    $request->input('limit') ?? 4,
                    Settings::get('vehicles-cross-sell-type')
                )
                ->toArray($request),
        ];
    }

    public function topTaxonomies(Request $request, VehicleRepository $repository): array
    {
        return collect($request->input('taxonomies', []))
            ->keyBy(fn($taxonomy) => $taxonomy)
            ->map(fn($taxonomy) => $this->cache("top-$taxonomy", 5, fn() => match ($taxonomy) {
                'makes' => $repository->topMakes($request->input('limit')),
                'body_types' => $repository->topBodyTypes($request->input('limit')),
                'fuel_types' => $repository->topFuelTypes($request->input('limit')),
                default => []
            }))
            ->filter()
            ->toArray();
    }
}
