<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;
use Mtc\MercuryDataModels\VehiclePrintLog;
use Tests\TestCase;

class VehiclePrintTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testPrintLogEmpty()
    {
        $vehicle = Vehicle::factory()->create([
            'price' => 123,
        ]);

        $this->assertEmpty($vehicle->latestPrintLog);
    }

    public function testPrintLogToBePrinted()
    {
        $vehicle = Vehicle::factory()->create([
            'price' => 123,
        ]);

        VehiclePrintLog::query()->create([
            'vehicle_id' => $vehicle->id,
            'price' => 456,
            'printed_at' => Carbon::now(),
        ]);

        $this->assertNotNull($vehicle->latestPrintLog);
        $this->assertEquals(456, $vehicle->latestPrintLog->price);
        $this->assertTrue($vehicle->toBePrinted);
    }

    public function testPrintLogNotToBePrinted()
    {
        $vehicle = Vehicle::factory()->create([
            'price' => 123,
        ]);

        VehiclePrintLog::query()->create([
            'vehicle_id' => $vehicle->id,
            'price' => 123,
            'printed_at' => Carbon::now(),
        ]);

        $this->assertNotNull($vehicle->latestPrintLog);
        $this->assertEquals(123, $vehicle->latestPrintLog->price);
        $this->assertFalse($vehicle->toBePrinted);
    }

    public function testSearch()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Vehicle Type',
            'config_key' => 'automotive-vehicle-filters-vehicle_type',
            'type' => 'boolean',
            'value' => true,
        ]);

        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
        ]);

        VehiclePrintLog::query()->create([
            'vehicle_id' => $vehicle->id,
            'price' => 123,
            'printed_at' => Carbon::now(),
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'));

        $response->assertStatus(200);

        $this->assertEquals(1, $response->json('results.total'));
        $this->assertCount(1, $response->json('results.data'));

        $this->assertArrayHasKey('to_be_printed', $response->json('results.data.0'));
        $this->assertArrayHasKey('dealership', $response->json('results.data.0'));
        $this->assertArrayHasKey('derivative', $response->json('results.data.0'));
        $this->assertArrayHasKey('finance_example', $response->json('results.data.0'));
        $this->assertArrayHasKey('fuel_type', $response->json('results.data.0'));
        $this->assertArrayHasKey('is_vat_applicable', $response->json('results.data.0'));
        $this->assertArrayHasKey('make', $response->json('results.data.0'));
        $this->assertArrayHasKey('manufacture_year', $response->json('results.data.0'));
        $this->assertArrayHasKey('model', $response->json('results.data.0'));
        $this->assertArrayHasKey('odometer_mi', $response->json('results.data.0'));
        $this->assertArrayHasKey('odometer_km', $response->json('results.data.0'));
        $this->assertArrayHasKey('original_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('previous_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('price', $response->json('results.data.0'));
        $this->assertArrayHasKey('rrp_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('title', $response->json('results.data.0'));
        $this->assertArrayHasKey('transmission', $response->json('results.data.0'));

        // assert that response contains search filter options
        $this->assertGreaterThan(0, $response->json('filters'));
        $this->assertArrayHasKey('vehicle_type', $response->json('filters'));
        $this->assertGreaterThan(0, $response->json('filters.vehicle_type.results'));
    }

    public function testFilters()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Price Min',
            'config_key' => 'automotive-vehicle-filters-price_min',
            'type' => 'boolean',
            'value' => true,
        ]);

        Vehicle::factory(2)->create([
            'is_published' => true,
            'price' => 50,
        ]);

        Vehicle::factory(3)->create([
            'is_published' => true,
            'price' => 100,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'), [
                'selections' => [
                    [
                        'type' => 'price_min',
                        'value' => 99,
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertEquals(3, $response->json('results.total'));
    }

    public function testToBePrintedFlag()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-price-asc',
            'type' => 'boolean',
            'value' => true,
        ]);

        $vehicle_same_price = Vehicle::factory()->create([
            'is_published' => true,
            'price' => 50,
        ]);

        $vehicle_different_price = Vehicle::factory()->create([
            'is_published' => true,
            'price' => 60,
        ]);

        Vehicle::factory()->create([
            'is_published' => true,
            'price' => 70,
        ]);

        // vehicle price has not changed since last print
        VehiclePrintLog::query()->create([
            'vehicle_id' => $vehicle_same_price->id,
            'price' => 50,
            'printed_at' => Carbon::now(),
        ]);

        // vehicle price has changed since last print
        VehiclePrintLog::query()->create([
            'vehicle_id' => $vehicle_different_price->id,
            'price' => 99,
            'printed_at' => Carbon::now(),
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'), [
                'sort_by' => 'price-asc',
            ]);

        $response->assertStatus(200);

        $this->assertEquals(3, $response->json('results.total'));
        $this->assertCount(3, $response->json('results.data'));

        $this->assertArrayHasKey('to_be_printed', $response->json('results.data.0'));
        $this->assertArrayHasKey('to_be_printed', $response->json('results.data.1'));
        $this->assertArrayHasKey('to_be_printed', $response->json('results.data.2'));

        // price did not change so should not be printed
        $this->assertFalse($response->json('results.data.0.to_be_printed'));

        // price did change so should be printed
        $this->assertTrue($response->json('results.data.1.to_be_printed'));

        // no previous print log so should be printed
        $this->assertTrue($response->json('results.data.2.to_be_printed'));
    }

    public function testLogBatchPrinted()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        $vehicles = Vehicle::factory(5)->create([
            'is_published' => true,
        ]);

        $vehicle_attribute = VehicleAttribute::query()
            ->firstOrCreate(
                [
                    'name' => 'lez_compliant',
                ],
                [
                    'slug' => 'lez_compliant',
                    'type' => 'boolean',
                    'validation' => [],
                    'data' => []
                ]
            );

        $vehicle_ids = collect($vehicles)->map(function ($vehicle) use ($vehicle_attribute) {
            VehicleAttributeValue::query()->create([
                'attribute_id' => $vehicle_attribute->id,
                'owner_id' => $vehicle->id,
                'owner_type' => 'vehicle',
                'value' => true,
                'value_integer' => 1,
                'slug' => $vehicle_attribute->slug,
                'type' => 'boolean'
            ]);

            return $vehicle->id;
        });

        $slugs = collect($vehicles)->map(function ($vehicle) {
            return $vehicle->slug;
        });

        // add a slug that will not be found to ensure the endpoint only returns found vehicles
        $slugs->add('test-slug-for-missing-vehicle');

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.log-batch-print'), [
                'vehicle_slugs' => $slugs,
            ]);

        $response->assertStatus(200);

        // assert that we get the expected data back in the response
        $this->assertCount(5, $response->json('vehicles'));
        $this->assertArrayHasKey('id', $response->json('vehicles.0'));
        $this->assertArrayHasKey('price', $response->json('vehicles.0'));
        $this->assertArrayHasKey('dealership', $response->json('vehicles.0'));
        $this->assertArrayHasKey('make', $response->json('vehicles.0'));
        $this->assertArrayHasKey('model', $response->json('vehicles.0'));
        $this->assertArrayHasKey('custom', $response->json('vehicles.0'));
        $this->assertArrayHasKey('lez_compliant', $response->json('vehicles.0.custom'));

        // assert that the expected vehicle print logs were created
        $this->assertCount(5, VehiclePrintLog::all());

        collect($vehicle_ids)->each(function ($vehicle_id) {
            $this->assertTrue(VehiclePrintLog::query()->where('vehicle_id', '=', $vehicle_id)->exists());
        });
    }

    public function testMiddlewarePrintingNotEnabled()
    {
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'));

        $response->assertStatus(403);
    }

    public function testMiddlewarePasswordRequiredMissing()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'));

        $response->assertStatus(401);
    }

    public function testMiddlewarePasswordRequiredIncorrect()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'), [
                'password' => 'bar'
            ]);

        $response->assertStatus(401);
    }

    public function testMiddlewarePasswordOk()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'bar',
            'config_key' => 'automotive-vehicle-printing-password',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicle-printing.search'), [
                'password' => 'foo'
            ]);

        $response->assertStatus(200);
    }
}
