<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Traits\HasPlaceholderImage;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Page;

class DealershipList extends JsonResource
{
    use HasPlaceholderImage;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        return [
            'dealerships' => $this->resource->map(fn (Dealership $dealership) => $this->mapDealership($dealership)),
            'franchises' => $this->franchiseNames(),
            'cities' => $this->cities(),
            'page' => new PageResource(Page::query()->where('slug', 'locations')->first()),
        ];
    }

    public function mapDealership(Dealership $dealership)
    {
        $coords = explode(',', $dealership->coordinates);
        return [
            'name' => $dealership->name,
            'address' => collect([
                $dealership->address1,
                $dealership->address2,
                $dealership->city,
                $dealership->postcode,
            ])->filter()->implode(', '),
            'city' => $dealership->city,
            'open' => $dealership->is_open,
            'openStart' => $dealership->opens_at,
            'openEnd' => $dealership->closes_at,
            'openDays' => [],
            'open_times' => $dealership->open_times,
            'alt_open_times' => Settings::get('dealership-second-open-times')
                ? $dealership->alt_open_times
                : [],
            'alt_open_time_name' => Settings::get('dealership-secondary-open-close-time-name'),
            'notes' => $dealership->notes,
            'images' => $this->getGallery($dealership),
            'coords' => [
                'lat' => $coords[0] ?? null,
                'lng' => $coords[1] ?? null,
            ],
            'url' => '/locations/' . $dealership->slug,
            'franchise' => $dealership->franchise,
        ];
    }

    protected function largeImageDimensions(): string
    {
        return 'full-width-banner';
    }

    protected function mediumImageDimensions(): string
    {
        return 'square';
    }

    protected function thumbnailImageDimensions(): string
    {
        return 'square';
    }

    private function cities(): array
    {
        return $this->resource->pluck('city')->unique()->filter()->toArray();
    }

    private function franchiseNames(): array
    {
        return $this->resource->pluck('franchise.name')->filter()->unique()->toArray();
    }
}
