<?php

namespace App;

use App\Http\Resources\VehicleList;
use App\Http\Resources\VehiclesCompareResource;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Mtc\MercuryDataModels\Vehicle;

class VehicleRepository
{
    public function similar(Vehicle $vehicle, int $limit = 4, string $matchType = 'price'): VehicleList
    {
        return new VehicleList(
            Vehicle::query()
                ->active()
                ->withRelationshipsForCardView()
                ->similar($matchType, $vehicle)
                ->where('id', 'not like', $vehicle->id)
                ->take($limit * 2)
                ->get()
                ->shuffle()
                ->take($limit)
        );
    }

    public function featured(?Vehicle $vehicle, int $limit = 4, ?array $filters = []): VehicleList
    {
        return new VehicleList(
            Vehicle::query()
                ->where('featured', 1)
                ->active()
                ->withRelationshipsForCardView()
                ->when($vehicle, fn(Builder $query) => $query->where('id', 'not like', $vehicle?->id))
                ->when(
                    $filters['make'] ?? null,
                    fn(Builder $query) => $query->whereHas(
                        'make.filterIndex',
                        fn(Builder $indexQuery) => $indexQuery->where('slug', $filters['make'])
                    )
                )
                ->when(
                    $filters['fuel_type'] ?? null,
                    fn(Builder $query) => $query->whereHas(
                        'fuelType.filterIndex',
                        fn(Builder $indexQuery) => $indexQuery->where('slug', $filters['fuel_type'])
                    )
                )
                ->inRandomOrder()
                ->take($limit)
                ->get()
        );
    }

    public function recentlyViewed(?Vehicle $vehicle, array $vehicleSlugs = [], int $limit = 4): VehicleList
    {
        return new VehicleList(
            Vehicle::query()
                ->whereIn('slug', $vehicleSlugs)
                ->active()
                ->withRelationshipsForCardView()
                ->when($vehicle, fn(Builder $query) => $query->where('id', 'not like', $vehicle?->id))
                ->inRandomOrder()
                ->take($limit)
                ->get()
        );
    }

    public function compare(array $vehicleSlugs = []): VehiclesCompareResource
    {
        return new VehiclesCompareResource(
            Vehicle::query()
                ->whereIn('slug', $vehicleSlugs)
                ->active()
                ->with(['features', 'specs'])
                ->withRelationshipsForCardView()
                ->get()
        );
    }

    public function recentlyAdded(int $limit = 4, array $filter = []): VehicleList
    {
        return new VehicleList(
            Vehicle::query()
                ->where($filter)
                ->active()
                ->withRelationshipsForCardView()
                ->latest()
                ->take($limit)
                ->get()
        );
    }

    public function trackView(Vehicle $vehicle): void
    {
        $view = $vehicle->views()
            ->firstOrNew(['date' => Carbon::now()->format('Y-m-d')]);
        $view->hits++;
        $view->save();
    }

    public function trackQuickView(Vehicle $vehicle): void
    {
        $view = $vehicle->views()
            ->firstOrNew(['date' => Carbon::now()->format('Y-m-d')]);
        $view->quick_view_hits++;
        $view->save();
    }
}
