<?php

namespace App\Http\Controllers;

use App\Filter;
use App\Http\Requests\NewCarSearchRequest;
use App\Http\Resources\NewCarListResource;
use App\Http\Resources\NewCarResource;
use App\Traits\ModelsForMakeInSearch;
use App\Traits\VehicleOfferData;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\NewCar;

class NewCarController extends Controller
{
    use ModelsForMakeInSearch;
    use VehicleOfferData;

    public function search(NewCarSearchRequest $request, ?string $make): array
    {
        $makeModel = Filter::matchSlugToRecord('make', $make);
        $vehicleModel = $request->filled('model_id')
            ? Filter::matchSlugToRecord('model', $request->input('model_id'))
            : null;

        $cars = NewCar::query()
            ->with([
                'mediaUses.media',
                'make.filterIndex',
                'model.filterIndex',
            ])
            ->where('make_id', $makeModel->id)
            ->when($request->filled('body_type'), fn($query) => $query->whereHas(
                'bodyStyle.filterIndex',
                fn($query) => $query->where('slug', $request->input('body_type'))
            ))
            ->when($request->filled('fuel_type'), fn($query) => $query->whereHas(
                'fuelType.filterIndex',
                fn($query) => $query->where('slug', $request->input('fuel_type'))
            ))
            ->when($request->filled('franchise'), fn($query) => $query->whereHas(
                'franchise',
                fn($query) => $query->where('slug', $request->input('franchise'))
            ))
            ->whereModelSlug($request->input('model_id'))
            ->setSortBy($request->input('sort_by'))
            ->active()
            ->paginate();

        $models = $this->getModelsForMake($makeModel->id, 'newCars');
        $this->addOfferDataToModels($makeModel->id, $models);

        return [
            'make' => $makeModel->name,
            'model' => $vehicleModel?->name,
            'models' => $models,
            'offers' => new NewCarListResource($cars),
            'content' => [
                'title' => __('automotive.new_cars_for_make', ['make' => $makeModel->name]),
                'description' => '',
            ]
        ];
    }

    public function show(NewCar $car): NewCarResource
    {
        return new NewCarResource($car);
    }

    private function addOfferDataToModels(int $make_id, Collection &$models): Collection
    {
        $model_offer_data = $this->getOfferSupportingData($make_id);
        $models = $models->map(function ($model) use ($model_offer_data) {
            if (empty($model['id'])) {
                $model['offer_data'] = $model_offer_data['all'];
            } else {
                $model['offer_data'] = $model_offer_data[$model['id']] ?? [];
            }

            return $model;
        });

        return $models;
    }
}
