<?php

namespace App\Http\Controllers;

use App\Facades\Feature;
use App\Filter;
use App\Http\Requests\NewCarSearchRequest;
use App\Http\Requests\ViewNewCarRequest;
use App\Http\Resources\NewCarListResource;
use App\Http\Resources\NewCarResource;
use App\Modules\PlaceholderImages\ImaginStudio;
use App\Traits\ModelsForMakeInSearch;
use Mtc\MercuryDataModels\NewCar;

class NewCarController extends Controller
{
    use ModelsForMakeInSearch;

    public function search(NewCarSearchRequest $request, ?string $make = null): array
    {
        $makeModel = $make ? Filter::matchSlugToRecord('make', $make) : null;
        $vehicleModel = $request->filled('model_id')
            ? Filter::matchSlugToRecord('model', $request->input('model_id'))
            : null;

        $cars = NewCar::query()
            ->with([
                'mediaUses.media',
                'make.filterIndex',
                'model.filterIndex',
                'fuelTypes.filterIndex',
            ])
            ->when($makeModel, fn($query) => $query->where('make_id', $makeModel->id))
            ->when($request->filled('body_type'), fn($query) => $query->whereHas(
                'bodyStyle.filterIndex',
                fn($query) => $query->where('slug', $request->input('body_type'))
            ))
            ->when($request->filled('fuel_type'), fn($query) => $query->whereHas(
                'fuelTypes.filterIndex',
                fn($query) => $query->whereIn('slug', $request->input('fuel_type'))
            ))
            ->when($request->filled('franchise'), fn($query) => $query->whereHas(
                'franchise',
                fn($query) => $query->where('slug', $request->input('franchise'))
            ))
            ->whereModelSlug($request->input('model_id'))
            ->setSortBy($request->input('sort_by'))
            ->active()
            ->paginate($request->input('per_page', 15));

        return [
            'make' => $makeModel?->name,
            'make_slug' => $makeModel?->slug,
            'model' => $vehicleModel?->name,
            'model_slug' => $vehicleModel?->slug,
            'models' => $makeModel ? $this->getModelsForMake($makeModel->id, 'newCars') : [],
            'offers' => new NewCarListResource($cars),
            'content' => [
                'title' => $makeModel
                    ? __('automotive.new_cars_for_make', ['make' => $makeModel->name])
                    : __('automotive.new_cars'),
                'description' => '',
            ],
            'imagin_studio_base_url' => Feature::isEnabled('imagin-studio-placeholders')
                ? (new ImaginStudio())->getBaseUrl()
                : null,
        ];
    }

    public function show(
        ViewNewCarRequest $request,
        NewCar $car
    ): NewCarResource {
        return new NewCarResource($car);
    }
}
