<?php

namespace App\Sales;

use App\Facades\Settings;
use App\Traits\FormatAsCurrency;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Vehicle;

class SilverBullet
{
    use FormatAsCurrency;

    public function getVehicleAction(Vehicle $vehicle, string $dealType = '')
    {
        if (!empty($vehicle->dealership->data['silver-bullet-disabled'])) {
            return [];
        }

        return [
            'type' => 'component',
            'value' => 'CatalogSilverBulletDeal',
            'data' => [
                'action' => $dealType,
                'title' => $this->getTitle($vehicle, $dealType),
                'url' => $this->url(Settings::get('sales-silver-bullet-domain'), $dealType),
                'id' => $vehicle->id,
                'vehicle_id' => $vehicle->uuid ?? $vehicle->id,
                'dealer_id' => $vehicle->dealership_id,
                'utm_source' => Settings::get('sales-silver-bullet-domain'),
                'utm_medium' => Settings::get('sales-silver-bullet-medium'),
                'utm_campaign' => Settings::get('sales-silver-bullet-campaign-name'),
                'vehicle_url' => '',
            ]
        ];
    }

    /**
     * Deal building url. Determined by type
     *
     * @param string|null $domain
     * @param string $dealType
     * @return string
     */
    private function url(?string $domain, string $dealType): string
    {
        return match ($dealType) {
            'finance' => rtrim($domain, '/') . '/start-finance/',
            'valuation' => rtrim($domain, '/') . '/start-trade-in/',
            default => rtrim($domain, '/') . '/start/',
        };
    }

    private function getTitleSetting(string $dealType)
    {
        return match ($dealType) {
            'finance' => 'sales-silver-bullet-finance-button-text',
            'valuation' => 'sales-silver-bullet-trade-in-button-text',
            'reserve' => 'sales-silver-bullet-reserve-button-text',
            default => 'sales-silver-bullet-deal-button-text',
        };
    }

    private function getTitle(Vehicle $vehicle, string $dealType)
    {
        $title = Settings::get($this->getTitleSetting($dealType));
        if ($dealType === 'reserve') {
            $cost = $vehicle->dealership->data['silver-bullet-reserve-cost'] ??
                Settings::get('sales-silver-bullet-reserve-cost');
            $cost = $cost ? $this->asCurrency($cost) : '';
            $title = str_replace(':amount', $cost, $title);
        }

        return $title;
    }
}
