<?php

namespace App\Traits;

use App\Facades\Feature;
use Illuminate\Database\Eloquent\Model;
use App\Modules\PlaceholderImages\ImaginStudio;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Setting;

trait HasPlaceholderImage
{
    abstract protected function largeImageDimensions(): string;
    abstract protected function mediumImageDimensions(): string;

    protected function getGallery(Model $model): Collection
    {
        if (is_null($model->mediaUses) || $model->mediaUses->isEmpty()) {
            return $this->getPlaceholderForGallery($model);
        }

        return $this->resource->mediaUses->map(fn($mediaUse) => [
            'title' => $mediaUse->title,
            'alt_text' => $mediaUse->alt_text,
            'description' => $mediaUse->description,
            'caption' => $mediaUse->caption,
            'src_large' => $mediaUse->getUrl($this->largeImageDimensions()),
            'src_medium' => $mediaUse->getUrl($this->mediumImageDimensions()),
            'type' => $mediaUse->media->type
        ]);
    }

    protected function getPreviewImage(Model $model): string
    {
        return $model->getPreviewImage($this->mediumImageDimensions())
            ?? $this->getPlaceholder($model, $this->mediumImageDimensions());
    }

    protected function getPlaceholderForGallery(Model $model): Collection
    {
        if ($this->canUseAdvancedPlaceholder($model)) {
            return $this->advancedPlaceholderGallery($model->getMorphClass(), $model)
                ->map(fn($image) => [
                    'alt_text' => '',
                    'src_large' => $image,
                    'src_medium' => $image,
                ]);
        }

        $image = $this->basicPlaceholderImage($model->getMorphClass());
        return collect([
            [
                'alt_text' => '',
                'src_large' => $image,
                'src_medium' => $image,
            ]
        ]);
    }

    protected function getPlaceholder(Model $model): string
    {
        if ($this->canUseAdvancedPlaceholder($model)) {
            return $this->advancedPlaceholderImage($model->getMorphClass(), $model);
        }

        return $this->basicPlaceholderImage($model->getMorphClass());
    }

    protected function advancedPlaceholderForGallery(string $type)
    {
        return collect(match ($type) {
            'vehicle' => [
                [
                    'src_large' => (new ImaginStudio())->getForVehicle($this->resource),
                    'src_medium' => (new ImaginStudio())->getForVehicle($this->resource),
                ]
            ],
            'offer' => [
                [
                    'src_large' => (new ImaginStudio())->getForOffer($this->resource),
                    'src_medium' => (new ImaginStudio())->getForOffer($this->resource),
                ]
            ],
            default => [],
        });
    }

    protected function advancedPlaceholderImage(string $type, Model $model): string
    {
        return match ($type) {
            'vehicle' => (new ImaginStudio())->getForVehicle($model),
            'new-car' => (new ImaginStudio())->getForNewCar($model),
            'offer' => (new ImaginStudio())->getForOffer($model),
            default => '',
        };
    }

    protected function advancedPlaceholderGallery(string $type, Model $model): Collection
    {
        return match ($type) {
            'vehicle' => $this->angles()->map(fn($i) => (new ImaginStudio())->getForVehicle($model, $i)),
            'new-car' => $this->angles()->map(fn($i) => (new ImaginStudio())->getForNewCar($model, $i)),
            'offer' => $this->angles()->map(fn($i) => (new ImaginStudio())->getForOffer($model, $i)),
            default => collect(),
        };
    }

    protected function basicPlaceholderImage(string $type): string
    {
        $dimensions = config(config("automotive.placeholder_media_sizes.placeholder-image-$type"));
        $mediaUses = Setting::query()->where('config_key', 'placeholder-image-' . $type)
            ->first()
            ?->mediaUses;

        return $mediaUses && $mediaUses->count() ? $mediaUses->first()->getUrl($dimensions['large']) : '';
    }

    protected function canUseAdvancedPlaceholder(Model $model): bool
    {
        return match ($model->getMorphClass()) {
            'vehicle', 'offer', 'new-car' => Feature::isEnabled('imagin-studio-placeholders'),
            default => false,
        };
    }

    protected function angles(): Collection
    {
        // https://docs.imagin.studio/guides/getting-our-images/selecting-angles
        return collect([
            1,
            5,
            9,
            13,
            17,
            21,
            22,
            23,
        ]);
    }
}
