<?php

namespace App\Traits;

use App\Facades\Settings;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Models\MediaSize;
use Mtc\MercuryDataModels\Contracts\ContentModel;
use Mtc\MercuryDataModels\Country;
use Mtc\MercuryDataModels\SeoDefault;

trait RetrievesContent
{
    protected Collection $mediaSizes;

    private function getContent(Collection $contentList): Collection
    {
        return $contentList->map(
            fn(ContentModel $content) => $this->isGlobalContent($content)
                ? $this->mapGlobalContent($content)
                : $this->mapContentElement($content)
        );
    }

    private function isGlobalContent(ContentModel $content): bool
    {
        return !empty($content->global_content_id);
    }

    private function mapGlobalContent(ContentModel $content): array
    {
        return [
            'ui_component' => $content->globalContent?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => [],

        ];
    }

    private function mapContentElement(ContentModel $content): array
    {
        return [
            'ui_component' => $content->contentElement?->ui_component ?? $this->componentFromField($content),
            'name' => $content->name,
            'slug' => $content->slug,
            'value' => $this->mapValue($content),
            'config' => [],
            'data' => $content->subContent && $this->componentHasUiResource($content) === false
                ? $this->getContent($content->subContent)
                : [],
        ];
    }

    private function mapValue(ContentModel $content)
    {
        if ($this->componentHasUiResource($content)) {
            $contentResource = $this->getUiResource($content);
            return new $contentResource($content);
        }

        if (in_array($content->data['fieldId'] ?? '', ['image', 'images'])) {
            return collect($content->mediaUses)
                ->map(fn($mediaUse) => [
                    'src' => $mediaUse->getUrl('square-tile'),
                    'sizes' => $this->allSizesForUse($content, $mediaUse),
                    'alt' => $mediaUse->alt_text,
                    'caption' => $mediaUse->caption,
                    'title' => $mediaUse->title,
                ]);
        }

        return $content->content;
    }

    private function componentHasUiResource(ContentModel $content): bool
    {
        if (!empty($content->global_content_id)) {
            return !empty($content->globalContent->ui_component)
                && class_exists(self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component);
        }

        return !empty($content->contentElement->ui_component)
            && class_exists(self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component);
    }

    private function getUiResource(ContentModel $content): string
    {
        return
            !empty($content->global_content_id)
                ? self::UI_RESOURCE_PREFIX . $content->globalContent->ui_component
                : self::UI_RESOURCE_PREFIX . $content->contentElement->ui_component;
    }

    private function componentFromField(ContentModel $content)
    {
        return $content->data['uiComponent'] ?? $this->primitiveField($content);
    }

    private function primitiveField(ContentModel $content)
    {
        return match ($content->data['fieldId'] ?? null) {
            'text-content' => 'UiArticleText',
            'text_area' => 'UiArticleText',
            'images' => 'UiArticleBanner',
            'image' => 'UiArticleBanner',
            default => null,
        };
    }

    private function fillSeo(string $section): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $default = SeoDefault::query()->where('section', $section)->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' =>  $this->resource->title ?? $this->resource->name,
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' =>  $this->resource->title ?? $this->resource->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{CONTENT_EXCERPT}}' => Str::limit($this->resource->description, 150),

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }


    private function getContactNumber(): ?array
    {
        if ($this->resource->dealership_id === null) {
            return null;
        }

        if ($this->resource->dealership->contact_no === null) {
            return null;
        }

        $countryCode = $this->resource->dealership->country;
//        $prefix = Cache::rememberForever(
//            'country-dial-code-' . $countryCode,
//            fn () => Country::query()->where('code', $countryCode)->first()?->dial_code
//        );

        $prefix = '+' . Country::query()->where('code', $countryCode)->first()?->dial_code;
        return [
            'number' => $prefix . $this->resource->dealership->contact_no,
            'display' => $prefix . ' ' . $this->displayNumber($this->resource->dealership->contact_no),
        ];
    }

    private function allSizesForUse(ContentModel $content, MediaUse $mediaUse): Collection
    {
        if (!isset($this->mediaSizes)) {
            $this->mediaSizes = MediaSize::query()
                ->where('model', $content->getMorphClass())
                ->get();
        }
        return $this->mediaSizes
            ->keyBy('label')
            ->map(fn(MediaSize $size) => $mediaUse->getUrl($size->label));
    }
}
