<?php

namespace Tests\Feature;

use App\Facades\Feature;
use App\Facades\Settings;
use App\Http\Resources\VehicleResource;
use App\Models\FilterIndex;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;
use Mtc\MercuryDataModels\VehicleAutoTraderData;
use Mtc\MercuryDataModels\VehicleFeature;
use Mtc\MercuryDataModels\VehicleFinance;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleStandardEquipment;
use Mtc\MercuryDataModels\VehicleView;
use Tests\TestCase;

class VehicleControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFilterLoads()
    {
        Vehicle::factory(10)->create([
            'is_published' => true,
        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('sort_options', $response->json());
        $this->assertArrayHasKey('results', $response->json());
        $this->assertArrayHasKey('filters', $response->json());
        $this->assertEquals(10, $response->json('results.total'));
    }

    public function testViewSingleVehicle()
    {
        Http::fake([
            'https://cdn.imagin.studio/getPaints*' => Http::response(['paintData' => ['paintCombinations' => []]])
        ]);
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $bodystyle = BodyStyleType::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
            'body_style_id' => $bodystyle->id
        ]);
        $vehicle->original_price = 123123;
        $vehicle->rrp_price = 456456;
        $vehicle->save();

        VehicleStandardEquipment::factory()->create([
            'price' => 0,
            'vat_amount' => 0,
            'vehicle_id' => $vehicle->id,
            'vehicle_type' => 'vehicle'
        ]);

        VehicleStandardEquipment::factory()->create([
            'price' => 0,
            'vat_amount' => 0,
            'vehicle_id' => $vehicle->id,
            'vehicle_type' => 'vehicle'
        ]);

        VehicleStandardEquipment::factory()->create([
            'price' => 100,
            'vat_amount' => 83.34,
            'vehicle_id' => $vehicle->id,
            'vehicle_type' => 'vehicle'
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('age_identifier', $response->json('data'));
        $this->assertArrayHasKey('battery_capacity_kwh', $response->json('data'));
        $this->assertArrayHasKey('battery_charge_time', $response->json('data'));
        $this->assertArrayHasKey('battery_quick_charge_time', $response->json('data'));
        $this->assertArrayHasKey('battery_range', $response->json('data'));
        $this->assertArrayHasKey('battery_usable_capacity_kwh', $response->json('data'));
        $this->assertArrayHasKey('bhp', $response->json('data'));
        $this->assertArrayHasKey('bodyStyle', $response->json('data'));
        $this->assertArrayHasKey('co2', $response->json('data'));
        $this->assertArrayHasKey('colour', $response->json('data'));
        $this->assertArrayHasKey('content', $response->json('data'));
        $this->assertArrayHasKey('dealership', $response->json('data'));
        $this->assertArrayHasKey('derivative', $response->json('data'));
        $this->assertArrayHasKey('door_count', $response->json('data'));
        $this->assertArrayHasKey('drivetrain', $response->json('data'));
        $this->assertArrayHasKey('engine_size_cc', $response->json('data'));
        $this->assertArrayHasKey('equipment', $response->json('data'));
        $this->assertArrayHasKey('features', $response->json('data'));
        $this->assertArrayHasKey('financeExamples', $response->json('data'));
        $this->assertArrayHasKey('fuel_type', $response->json('data'));
        $this->assertArrayHasKey('gross_vehicle_weight_kg', $response->json('data'));
        $this->assertArrayHasKey('main_video_url', $response->json('data'));
        $this->assertArrayHasKey('make', $response->json('data'));
        $this->assertArrayHasKey('manufacture_year', $response->json('data'));
        $this->assertArrayHasKey('model', $response->json('data'));
        $this->assertArrayHasKey('mpg', $response->json('data'));
        $this->assertArrayHasKey('odometer_km', $response->json('data'));
        $this->assertArrayHasKey('odometer_mi', $response->json('data'));
        $this->assertArrayHasKey('original_price', $response->json('data'));
        $this->assertArrayHasKey('payload_kg', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
        $this->assertArrayHasKey('registration_number', $response->json('data'));
        $this->assertArrayHasKey('rrp_price', $response->json('data'));
        $this->assertArrayHasKey('seats', $response->json('data'));
        $this->assertArrayHasKey('specs', $response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('transmission', $response->json('data'));
        $this->assertArrayHasKey('type', $response->json('data'));
        $this->assertArrayHasKey('wheelbase_mm', $response->json('data'));
        $this->assertArrayHasKey('wheelbase_type', $response->json('data'));

        $this->assertNotNull($response->json('data.age_identifier'));
        $this->assertNotNull($response->json('data.battery_capacity_kwh'));
        $this->assertNotNull($response->json('data.battery_quick_charge_time'));
        $this->assertNotNull($response->json('data.battery_range'));
        $this->assertNotNull($response->json('data.bodyStyle'));
        $this->assertNotNull($response->json('data.co2'));
        $this->assertNotNull($response->json('data.colour'));
        $this->assertNotNull($response->json('data.content'));
        $this->assertNotNull($response->json('data.dealership'));
        $this->assertNotNull($response->json('data.derivative'));
        $this->assertNotNull($response->json('data.door_count'));
        $this->assertNotNull($response->json('data.drivetrain'));
        $this->assertNotNull($response->json('data.engine_size_cc'));
        $this->assertNotNull($response->json('data.equipment'));
        $this->assertNotNull($response->json('data.features'));
        $this->assertNotNull($response->json('data.financeExamples'));
        $this->assertNotNull($response->json('data.fuel_type'));
        $this->assertNotNull($response->json('data.main_video_url'));
        $this->assertNotNull($response->json('data.make'));
        $this->assertNotNull($response->json('data.manufacture_year'));
        $this->assertNotNull($response->json('data.model'));
        $this->assertNotNull($response->json('data.mpg'));
        $this->assertNotNull($response->json('data.odometer_km'));
        $this->assertNotNull($response->json('data.odometer_mi'));
        $this->assertNotNull($response->json('data.original_price'));
        $this->assertNotNull($response->json('data.price'));
        $this->assertNotNull($response->json('data.registration_number'));
        $this->assertNotNull($response->json('data.rrp_price'));
        $this->assertNotNull($response->json('data.seats'));
        $this->assertNotNull($response->json('data.specs'));
        $this->assertNotNull($response->json('data.title'));
        $this->assertNotNull($response->json('data.transmission'));
        $this->assertNotNull($response->json('data.type'));

        $this->assertArrayHasKey('extra', $response->json('data'));

        $this->assertArrayHasKey('saving_total', $response->json('data.extra'));

        $this->assertCount(3, $response->json('data.equipment'));
    }

    public function testQuickViewVehicle()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        SeoDefault::query()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.quick-view', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertArrayHasKey('title', $response->json('data'));
        $this->assertArrayHasKey('price', $response->json('data'));
        $this->assertArrayHasKey('actions', $response->json('data'));
        $this->assertArrayHasKey('odometer_mi', $response->json('data'));
        $this->assertArrayHasKey('odometer_km', $response->json('data'));
        $this->assertArrayHasKey('fuel_type', $response->json('data'));
        $this->assertArrayHasKey('transmission', $response->json('data'));

        $this->assertTrue(VehicleView::query()->where('vehicle_id', $vehicle->id)->where('quick_view_hits', 1)->exists());
    }

    public function testViewVehicleWithAccessKey()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => false,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));
        $response->assertStatus(403);
        $response = $this->asTenant(tenant())->getJson(
            route('vehicles.show', 'foo-baz-bar') . '?a=' . base64_encode($vehicle->id . '-' . $vehicle->slug)
        );
        $response->assertStatus(200);
    }

    public function testRecentlyViewed()
    {
        $vehicles = Vehicle::factory(3)->create(['is_published' => true]);
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.recentlyViewed', 'foo-baz-bar') . '?'
                . http_build_query(['slugs' => $vehicles->pluck('slug')->toArray()]));

        $response->assertStatus(200);
        $this->assertArrayHasKey('items', $response->json('recently_viewed'));
        $this->assertCount(3, $response->json('recently_viewed.items'));
    }

    public function testViewWithDefaultFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        VehicleFinance::factory()->create(['vehicle_id' => $vehicle->id]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertEquals('FinanceCalculator', $response->json('data.finance.value'));
        $this->assertArrayHasKey('change_values', $response->json('data.finance.data'));
        $this->assertArrayHasKey('values', $response->json('data.finance.data'));
    }

    public function testViewWithIvendiFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-ivendi-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('IVendiCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithCodeweaversFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-codeweavers-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('CodeweaversCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testViewWithEvolutionFinanceWidget()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-evolution-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'finance-term',
            'value' => 60,
            'type' => 'string',
        ]);
        $dealership = Dealership::factory()->create();
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'slug' => 'foo-baz-bar',
            'dealership_id' => $dealership->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('vehicles.show', 'foo-baz-bar'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('finance', $response->json('data'));
        $this->assertIsArray($response->json('data.finance'));
        $this->assertEquals('component', $response->json('data.finance.type'));
        $this->assertEquals('EvolutionCalculator', $response->json('data.finance.value'));
        $this->assertEquals('60', $response->json('data.finance.data.term'));
    }

    public function testVehicleResource()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'bool',
        ]);

        $vehicle_resource_array = (new VehicleResource(Vehicle::factory()->create()))->toArray(null);
        $extras = $vehicle_resource_array['extra'];
        $lez_compliant = collect($extras)->where('name', '=', 'lez_compliant');

        $this->assertEmpty($lez_compliant);
    }

    public function testVehicleAttributes()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Vehicle Make'
        ]);

        $make->filterIndex()->create([
            'slug' => 'vehicle-make'
        ]);

        $model = VehicleModel::factory()->create([
            'name' => 'Vehicle Model',
            'make_id' => $make->id
        ]);

        $model->filterIndex()->create([
            'slug' => 'vehicle-model'
        ]);

        $vehicle = Vehicle::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);

        $this->assertNotEmpty($vehicle);

        // try setting attribute
        $vehicle_attribute = VehicleAttribute::query()
            ->firstOrCreate(
                [
                    'name' => 'lez_compliant',
                ],
                [
                    'slug' => 'lez_compliant',
                    'type' => 'boolean',
                    'validation' => [],
                    'data' => []
                ]
            );

        $vehicle_attribute_coming_soon = VehicleAttribute::query()
            ->firstOrCreate(
                [
                    'name' => 'coming_soon',
                ],
                [
                    'slug' => 'coming_soon',
                    'type' => 'boolean',
                    'validation' => [],
                    'data' => []
                ]
            );

        // assert that vehicle doesn't have any attributes
        $example_vehicle_lez_compliant = $vehicle->attributeValues()->where('attribute_id', '=', $vehicle_attribute->id)->first();

        $this->assertEmpty($example_vehicle_lez_compliant);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'HP'
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'PCP'
        ]);

        // add an attribute value for the vehicle
        $vehicle_attribute_value = VehicleAttributeValue::query()->updateOrCreate(
            [
                'attribute_id' => $vehicle_attribute->id,
                'vehicle_id' => $vehicle->id,
            ],
            [
                'value' => true,
                'value_integer' => 1,
                'slug' => $vehicle_attribute->slug,
                'type' => 'boolean'
            ]
        );

        VehicleAttributeValue::query()->updateOrCreate(
            [
                'attribute_id' => $vehicle_attribute_coming_soon->id,
                'vehicle_id' => $vehicle->id,
            ],
            [
                'value' => false,
                'value_integer' => 0,
                'slug' => $vehicle_attribute_coming_soon->slug,
                'type' => 'boolean'
            ]
        );

        $this->assertEquals(2, VehicleAttributeValue::count());
        $this->assertEquals($vehicle->id, $vehicle_attribute_value->vehicle_id);
        $this->assertEquals($vehicle_attribute->id, $vehicle_attribute_value->attribute_id);
        $this->assertGreaterThan(0, $vehicle->attributeValues->count());

        // check that looking up the vehicle attribute works as expected
        $example_vehicle_lez_compliant = $vehicle->attributeValues()
            ->whereHas('attribute', function ($query) {
                $query->where('name', '=', 'lez_compliant');
            })->first();

        $this->assertNotEmpty($example_vehicle_lez_compliant);
        $this->assertEquals('lez_compliant', $example_vehicle_lez_compliant->attribute->name);
        $this->assertEquals(1, $example_vehicle_lez_compliant->value);

        // ensure that asking for values for missing vehicle attributes is handled gracefully
        $test_missing_value = $vehicle->attributeValues()
            ->whereHas('attribute', function ($query) {
                $query->where('name', '=', 'missing_value');
            })->first();

        $this->assertNull($test_missing_value);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => $vehicle->id,
            'price_point' => 'Good'
        ]);

        $vehicle_resource_array = (new VehicleResource($vehicle))->toArray(null);
        $extras = $vehicle_resource_array['extra'];
        $lez_compliant = collect($extras)->where('name', '=', 'lez_compliant');
        $missing_value = collect($extras)->where('name', '=', 'my_missing_value');

        $this->assertNotEmpty($lez_compliant);
        $this->assertEmpty($missing_value);

        $this->assertArrayHasKey('autoTraderData', $vehicle_resource_array);
        $this->assertNotEmpty($vehicle_resource_array['autoTraderData']);
        $this->assertEquals('Good', $vehicle_resource_array['autoTraderData']['price_point']);

        $this->assertArrayHasKey('custom', $vehicle_resource_array);
        $this->assertArrayHasKey('lez_compliant', $vehicle_resource_array['custom']);
        $this->assertNotNull($vehicle_resource_array['custom']['lez_compliant']);
        $this->assertTrue($vehicle_resource_array['custom']['lez_compliant']);

        $this->assertArrayHasKey('coming_soon', $vehicle_resource_array['custom']);
        $this->assertFalse($vehicle_resource_array['custom']['coming_soon']);

        $this->assertEquals('vehicle-make', $vehicle_resource_array['make']['slug']);
        $this->assertEquals('vehicle-model', $vehicle_resource_array['model']['slug']);

        $this->assertNotNull($vehicle_resource_array['financeExamples']);
        $this->assertCount(2, $vehicle_resource_array['financeExamples']);
    }

    public function testVehicleDealershipFranchise()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'valuation-auto-trader-enabled',
            'value' => false,
            'type' => 'boolean',
        ]);

        $franchise = Franchise::factory()->create([
            'name' => 'foo'
        ]);

        $dealership = Dealership::factory()->create([
            'franchise_id' => $franchise->id
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id
        ]);

        $this->assertEquals($franchise->name, $vehicle->dealership->franchise->name);

        $vehicle_resource_array = (new VehicleResource($vehicle))->toArray(null);

        $this->assertArrayHasKey('franchise', $vehicle_resource_array['dealership']);
        $this->assertEquals($franchise->name, $vehicle_resource_array['dealership']['franchise']['name']);
    }
}
