<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAutoTraderData;
use Mtc\MercuryDataModels\VehicleFinance;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\FilterSettingSeed;
use Tests\TestCase;

class VehicleFiltersTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFilterByMake()
    {
        $this->seed();
        $this->seed(FilterSettingSeed::class);
        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'make_id' => $make->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'make_id' => $make2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'make_id' => 100,
            ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First Make
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second make
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both Makes
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->id,
                    ],
                    [
                        'type' => 'make',
                        'value' => $make2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByModel()
    {
        $this->seed(FilterSettingSeed::class);
        $model = VehicleModel::factory()->create();
        $model2 = VehicleModel::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'model_id' => $model->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'model_id' => $model2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'model_id' => 100,
        ]);

        // No  specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model->id,
                    ],
                    [
                        'type' => 'model',
                        'value' => $model2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByAge()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'manufacture_year' => 2021,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'manufacture_year' => 2015,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'manufacture_year' => 2009,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => 2020,
                    ],
                    [
                        'type' => 'age',
                        'value' => 2021,
                    ],
                    [
                        'type' => 'age',
                        'value' => 2022,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => 2015,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => 2021
                    ],
                    [
                        'type' => 'age',
                        'value' => 2015
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(8, $response3->json('results.total'));
    }

    public function testFilterByBodyStyle()
    {
        $this->seed(FilterSettingSeed::class);
        $body_style = BodyStyleType::factory()->create();
        $body_style2 = BodyStyleType::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'body_style_id' => $body_style->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'body_style_id' => $body_style2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'body_style_id' => 100,
        ]);

        // No specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style->id,
                    ],
                    [
                        'type' => 'body_type',
                        'value' => $body_style2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByColour()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'colour' => 'red',
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'colour' => 'black',
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'colour' => 'silver',
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'black',
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'black',
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }


    public function testFilterByDoorCount()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'door_count' => 4,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'door_count' => 5,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'door_count' => 3,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 4,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 5,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 4,
                    ],
                    [
                        'type' => 'door_count',
                        'value' => 5,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByEngineSize()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'engine_size_cc' => 1400,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'engine_size_cc' => 1999,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'engine_size_cc' => 2500,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' =>  1500,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' => 2000,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(8, $response3->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' =>  2000,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(8, $response3->json('results.total'));
    }

    public function testFilterByFuelType()
    {
        $this->seed(FilterSettingSeed::class);
        $fuel_type = FuelType::factory()->create();
        $fuel_type2 = FuelType::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'fuel_type_id' => $fuel_type->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'fuel_type_id' => $fuel_type2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'fuel_type_id' => 100,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First Make
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type->id,
                    ],
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByLocation()
    {
        $this->seed(FilterSettingSeed::class);
        $location = Dealership::factory()->create();
        $location2 = Dealership::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'dealership_id' => $location->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'dealership_id' => $location2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'dealership_id' => 100,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location->id,
                    ],
                    [
                        'type' => 'location',
                        'value' => $location2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByMileage()
    {
        $this->seed(FilterSettingSeed::class);
        Settings::make([
            'tab' => 'foo',
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'Distance measurement',
            'config_key' => 'automotive-distance_measurement',
            'type' => 'string',
            'value' => 'mi',
        ]);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'odometer_mi' => 60000,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'odometer_mi' => 45000,
        ]);
        Vehicle::factory()->create([
            'is_published' => true,
            'odometer_mi' => 48000,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'odometer_mi' => 25000,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(19, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' =>  30000,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(10, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' => 47500,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(13, $response3->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' => 50000,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(14, $response3->json('results.total'));
    }

    public function testFilterByPrice()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'price' => 10200,
            'monthly_price' => 250,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'price' => 8500,
            'monthly_price' => 170,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'price' => 5000,
            'monthly_price' => 130,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'full',
                            'min' => 10000,
                            'max' => 12500,
                        ],
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'full',
                            'min' => 7500,
                            'max' => 10000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

    }

    public function testFilterBySeatCount()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'seats' => 4,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'seats' => 2,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'seats' => 5,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 4,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 2,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 4,
                    ],
                    [
                        'type' => 'seats',
                        'value' => 2,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByTransmission()
    {
        $this->seed(FilterSettingSeed::class);
        $transmission = TransmissionType::factory()->create();
        $transmission2 = TransmissionType::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'transmission_id' => $transmission->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'transmission_id' => $transmission2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'transmission_id' => 100,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterBySearchTerm()
    {
        $this->markTestSkipped('Unable to test fulltext search in sqlite');
    }

    public function testFilterByMultipleFilters()
    {
        $this->seed(FilterSettingSeed::class);
        $transmission = TransmissionType::factory()->create();
        $transmission2 = TransmissionType::factory()->create();
        Vehicle::factory(3)->create([
            'transmission_id' => $transmission->id,
            'colour' => 'blue',
            'is_published' => true,
        ]);
        Vehicle::factory(4)->create([
            'transmission_id' => $transmission->id,
            'colour' => 'red',
            'is_published' => true,
        ]);
        Vehicle::factory(5)->create([
            'transmission_id' => $transmission2->id,
            'colour' => 'blue',
            'is_published' => true,
        ]);
        Vehicle::factory(6)->create([
            'transmission_id' => $transmission2->id,
            'colour' => 'red',
            'is_published' => true,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'transmission_id' => 100,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(28, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(4, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'blue',
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(10, $response4->json('results.total'));
    }
    public function testFilterByConsumption()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create([
            'is_published' => true,
            'mpg' => 21,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'mpg' => 40,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'mpg' => 55,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'consumption',
                        'value' => 50,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(10, $response2->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'consumption',
                        'value' => 25,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(13, $response3->json('results.total'));

        // Second
        $response3 = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'consumption',
                        'value' => 20
                    ],
                    [
                        'type' => 'consumption',
                        'value' => 40
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(18, $response3->json('results.total'));
    }

    public function testFiltersAreEnabledBySettings()
    {
        $this->seed(FilterSettingSeed::class);
        $filters = App::make(FilterInstance::class)->getFilters();
        $this->assertEquals(16, $filters->count());

        Settings::update('automotive-vehicle-filters-colour', false);
        Settings::update('automotive-vehicle-filters-fuel_type', false);
        $filters = App::make(FilterInstance::class)->getFilters();
        $this->assertEquals(14, $filters->count());
        $this->assertArrayNotHasKey('colour', $filters);
        $this->assertArrayNotHasKey('fuel_type', $filters);
    }

    public function testFilteWidget()
    {
        $this->seed(FilterSettingSeed::class);
        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();
        Vehicle::factory(5)->create([
            'is_published' => true,
            'make_id' => $make->id,
        ]);
        Vehicle::factory(3)->create([
            'is_published' => true,
            'make_id' => $make2->id,
        ]);
        Vehicle::factory(10)->create([
            'is_published' => true,
            'make_id' => 100,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.widget'));

        $response->assertStatus(200);
        $this->assertTrue(count($response->json('filters')) > 1);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.widget'), [
                'only_filters' => [
                    'make'
                ]
            ]);
        $response->assertStatus(200);
        $this->assertArrayHasKey('make', $response->json('filters'));
    }

    public function testPricePointFilter()
    {
        $this->seed(FilterSettingSeed::class);


        // create some testing data

        Vehicle::factory()->create([
            'id' => 123,
            'is_published' => true,
        ]);

        Vehicle::factory()->create([
            'id' => 111,
            'is_published' => true
        ]);

        VehicleAutoTraderData::factory()->create([
           'vehicle_id' => 123,
           'price_point' => 'Fair'
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 111,
            'price_point' => 'Good'
        ]);

        // try search with no filters. Should bring back all of our example vehicles.
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        // test that call worked and brought back results
        $response->assertStatus(200);
        $this->assertEquals(2, $response->json('results.total'));

        $selected_price_point_filters = 0;

        foreach ($response->json('filters.price_point.results') as $filter_option) {
            if ($filter_option['selected']) {
                $selected_price_point_filters++;
            }
        }

        $this->assertEquals(0, $selected_price_point_filters);

        // search with a filter for one of the vehicles.
        // should return filtered vehicles.
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price_point',
                        'value' => 'Good'
                    ]
                ]
            ]);

        // assert that the price point filter results match what we expect
        $response->assertStatus(200);
        $this->assertEquals(1, $response->json('results.total'));

        $selected_price_point_filters = 0;

        foreach ($response->json('filters.price_point.results') as $filter_option) {
            if ($filter_option['selected']) {
                $selected_price_point_filters++;
            }
        }

        $this->assertEquals(1, $selected_price_point_filters);

        // filter on a price point that does not belong to any vehicles.
        // should not return vehicles.
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price_point',
                        'value' => 'Poor'
                    ]
                ]
            ]);

        // assert that the price point filter results match what we expect
        $response->assertStatus(200);
        $this->assertEquals(0, $response->json('results.total'));

        // test multiple filter selections
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price_point',
                        'value' => 'Good'
                    ],
                    [
                        'type' => 'price_point',
                        'value' => 'Fair'
                    ]
                ]
            ]);

        // assert that the price point filter results match what we expect
        $response->assertStatus(200);
        $this->assertEquals(2, $response->json('results.total'));

        $selected_price_point_filters = 0;

        foreach ($response->json('filters.price_point.results') as $filter_option) {
            if ($filter_option['selected']) {
                $selected_price_point_filters++;
            }
        }

        $this->assertEquals(2, $selected_price_point_filters);
    }

    public function testMinimumPricePointFilter()
    {
        $this->seed(FilterSettingSeed::class);

        // create our testing data
        Vehicle::factory()->create([
            'id' => 123,
            'is_published' => true,
        ]);

        Vehicle::factory()->create([
            'id' => 456,
            'is_published' => true
        ]);

        Vehicle::factory()->create([
            'id' => 789,
            'is_published' => true
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 123,
            'price_point' => 'Fair'
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 456,
            'price_point' => 'Good'
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 789,
            'price_point' => 'Great'
        ]);

        Settings::make([
            'tab' => 'automotive',
            'section' => 'AutoTrader',
            'group' => 'Enriched vehicle data',
            'name' => 'Display price indicator when value is equal or better than',
            'config_key' => 'autotrader-show-price-indicator-min-value',
            'type' => 'string',
            'value' => 'Good',
            'min_tier' => 'pro',
            'order' => 22,
        ]);

        // try search with no filters. Should bring back all of our example vehicles.
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        // test that call worked and brought back results
        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('results.total'));

        // get the list of price point filters returned by the search
        $available_price_point_selections = [];

        foreach ($response->json('filters.price_point.results') as $filter_option) {
            $available_price_point_selections[$filter_option['name']] = $filter_option['name'];
        }

        // assert that unwanted values were not returned (case sensitive)
        $this->assertArrayHasKey('Good', $available_price_point_selections);
        $this->assertArrayHasKey('Great', $available_price_point_selections);
        $this->assertArrayNotHasKey('Fair', $available_price_point_selections);
    }

    public function testPricePointFilterOptionOrder()
    {
        $this->seed(FilterSettingSeed::class);

        // create our testing data
        Vehicle::factory()->create([
            'id' => 123,
            'is_published' => true,
        ]);

        Vehicle::factory()->create([
            'id' => 456,
            'is_published' => true
        ]);

        Vehicle::factory()->create([
            'id' => 789,
            'is_published' => true
        ]);

        Vehicle::factory()->create([
            'id' => 999,
            'is_published' => true
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 123,
            'price_point' => 'Fair'
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 456,
            'price_point' => 'Good'
        ]);

        VehicleAutoTraderData::factory()->create([
            'vehicle_id' => 789,
            'price_point' => 'Great'
        ]);

        Settings::make([
            'tab' => 'automotive',
            'section' => 'AutoTrader',
            'group' => 'Enriched vehicle data',
            'name' => 'Display price indicator when value is equal or better than',
            'config_key' => 'autotrader-show-price-indicator-min-value',
            'type' => 'string',
            'value' => 'Fair',
            'min_tier' => 'pro',
            'order' => 22,
        ]);

        // try search with no filters. Should bring back all of our example vehicles.
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        // test that call worked and brought back results
        $response->assertStatus(200);
        $this->assertEquals(4, $response->json('results.total'));

        // get the list of price point filters returned by the search
        $available_price_point_selections = [];

        foreach ($response->json('filters.price_point.results') as $filter_option) {
            $available_price_point_selections[] = $filter_option['name'];
        }

        $this->assertEquals(3, count($available_price_point_selections));

        $this->assertEquals('Great', $available_price_point_selections[0]);
        $this->assertEquals('Good', $available_price_point_selections[1]);
        $this->assertEquals('Fair', $available_price_point_selections[2]);
    }

    public function testVehicleData()
    {
        $this->seed();
        $this->seed(FilterSettingSeed::class);

        Vehicle::factory(5)->create([
            'is_published' => true,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('results.total'));

        $this->assertArrayHasKey('autoTraderData', $response->json('results.data.0'));
        $this->assertArrayHasKey('dealership', $response->json('results.data.0'));
        $this->assertArrayHasKey('derivative', $response->json('results.data.0'));
        $this->assertArrayHasKey('finance_example', $response->json('results.data.0'));
        $this->assertArrayHasKey('fuel_type', $response->json('results.data.0'));
        $this->assertArrayHasKey('make', $response->json('results.data.0'));
        $this->assertArrayHasKey('manufacture_year', $response->json('results.data.0'));
        $this->assertArrayHasKey('model', $response->json('results.data.0'));
        $this->assertArrayHasKey('odometer_mi', $response->json('results.data.0'));
        $this->assertArrayHasKey('odometer_km', $response->json('results.data.0'));
        $this->assertArrayHasKey('original_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('previous_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('price', $response->json('results.data.0'));
        $this->assertArrayHasKey('rrp_price', $response->json('results.data.0'));
        $this->assertArrayHasKey('title', $response->json('results.data.0'));
        $this->assertArrayHasKey('transmission', $response->json('results.data.0'));
        $this->assertArrayHasKey('type', $response->json('results.data.0'));
        $this->assertArrayHasKey('custom', $response->json('results.data.0'));
        $this->assertArrayHasKey('age_identifier', $response->json('results.data.0'));
    }

    public function testVehicleSavings()
    {
        $this->seed();
        $this->seed(FilterSettingSeed::class);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled-previous-price',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'vehicle-savings-type',
            'value' => 'peter-vardy',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'id' => 1,
            'location_stock' => 'VE',
            'name' => 'Carz Edinburgh'
        ]);

        Vehicle::factory()->create([
            'id' => 123,
            'is_published' => true,
            'type' => 'car',
            'price' => 8000,
            'original_price' => 8000,
            'previous_price' => 9000,
            'dealership_id' => $dealership->id,
        ]);

        // No makes specified
        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);
        $this->assertEquals(1, $response->json('results.total'));
        $this->assertEquals(1000, $response->json('results.data.0.extra.saving_original_price'));
    }

    public function testVehicleFinanceExample()
    {
        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'General',
            'name' => 'Load finance example for vehicle listing',
            'config_key' => 'vehicle-list-load-finance-example',
            'type' => 'boolean',
            'value' => true,
            'superadmin' => true,
            'min_tier' => 'standard',
            'description' => 'Enable loading finance example for lists of vehicles, such as search result page.<br/>',
            'order' => 0,
        ]);

        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'HP',
            'monthly_price' => 123
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'PCP',
            'monthly_price' => 456
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);

        $this->assertEquals('HP', $response->json('results.data.0.finance_example.finance_type'));
        $this->assertEquals(123, $response->json('results.data.0.finance_example.monthly_price'));
    }

    public function testVehicleFinanceDisabled()
    {
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'HP',
            'monthly_price' => 123
        ]);

        VehicleFinance::factory()->create([
            'vehicle_id' => $vehicle->id,
            'finance_type' => 'PCP',
            'monthly_price' => 456
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'));

        $response->assertStatus(200);

        $this->assertEmpty($response->json('results.data.0.finance_example'));
    }

    public function testVehicleResultFranchise()
    {
        $franchise = Franchise::factory()->create([
            'name' => 'foo'
        ]);

        $dealership = Dealership::factory()->create([
            'franchise_id' => $franchise->id
        ]);

        Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_published' => true,
            'price' => 123,
        ]);

        $dealership2 = Dealership::factory()->create();

        Vehicle::factory()->create([
            'dealership_id' => $dealership2->id,
            'is_published' => true,
            'price' => 456,
        ]);

        Vehicle::factory()->create([
            'is_published' => true,
            'price' => 789,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('vehicles.index'), [
                'sort_by' => 'price-asc',
            ]);

        $response->assertStatus(200);

        $this->assertEmpty($response->json('results.data.0.finance_example'));
        $this->assertEquals(3, $response->json('results.total'));
        $this->assertNotNull($response->json('results.data.0.dealership'));
        $this->assertEquals($franchise->name, $response->json('results.data.0.dealership.franchise.name'));
        $this->assertArrayNotHasKey('franchise', $response->json('results.data.1.dealership'));
    }
}
