<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\KeyloopLeaseResidualValue;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Tests\TestCase;

class KeyloopFinanceTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFinanceRequest()
    {
        $this->createSettings();

        $variant = KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 34766.33,
            'delivery_ex_vat_ex_vrt' => 0,
            'vehicle_vrt' => 0,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 10000,
            'residual_value_excluding_vat' => 5555,
            'residual_value_including_vat' => 5555 * 1.23,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 48,
            'annual_mileage' => 10000,
            'residual_value_excluding_vat' => 7777,
            'residual_value_including_vat' => 7777 * 1.23,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 15000,
            'residual_value_excluding_vat' => 11871.43,
            'residual_value_including_vat' => 11871.43 * 1.23,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('lease-vehicles.get-finance', $variant->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('36-15000-1', $response->json('data.finance_options.monthly_prices'));
        $this->assertEquals(771.57, $response->json('data.finance_options.monthly_prices.36-15000-1.monthly_price_ex_vat'));
        $this->assertEquals(949.03, $response->json('data.finance_options.monthly_prices.36-15000-1.monthly_price_inc_vat'));
    }

    public function testBulkFinance()
    {
        $this->createSettings();

        $variant_1 = KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 34766.33,
            'delivery_ex_vat_ex_vrt' => 0,
            'vehicle_vrt' => 0,
        ]);
        $variant_2 = KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 45000,
            'delivery_ex_vat_ex_vrt' => 0,
            'vehicle_vrt' => 0,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant_1->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 15000,
            'residual_value_excluding_vat' => 11871.43,
            'residual_value_including_vat' => 11871.43 * 1.23,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant_2->external_variant_id,
            'contract_length' => 24,
            'annual_mileage' => 10000,
            'residual_value_excluding_vat' => 31000.00,
            'residual_value_including_vat' => 31000.00 * 1.23,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('lease-vehicles.get-bulk-finance'), [
                'variant_slugs' => [
                    $variant_1->slug,
                    $variant_2->slug,
                ],
            ]);

        $response->assertStatus(200);
        $this->assertCount(2, $response->json('data.variants'));
        $this->assertArrayHasKey($variant_1->slug, $response->json('data.variants'));
        $this->assertArrayHasKey($variant_2->slug, $response->json('data.variants'));

        $price_key_1 = 'data.variants.' . $variant_1->slug . '.monthly_prices.36-15000-1';
        $this->assertEquals(771.57, $response->json($price_key_1 . '.monthly_price_ex_vat'));
        $this->assertEquals(949.03, $response->json($price_key_1 . '.monthly_price_inc_vat'));

        $price_key_2 = 'data.variants.' . $variant_2->slug . '.monthly_prices.24-10000-1';
        $this->assertEquals(802.82, $response->json($price_key_2 . '.monthly_price_ex_vat'));
        $this->assertEquals(987.46, $response->json($price_key_2 . '.monthly_price_inc_vat'));
    }

    public function testItemFinance()
    {
        $this->createSettings();

        $variant = KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 34766.33,
            'delivery_ex_vat_ex_vrt' => 0,
            'vehicle_vrt' => 0,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 10000,
            'residual_value_excluding_vat' => 5555,
            'residual_value_including_vat' => 5555 * 1.23,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 48,
            'annual_mileage' => 10000,
            'residual_value_excluding_vat' => 7777,
            'residual_value_including_vat' => 7777 * 1.23,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 15000,
            'residual_value_excluding_vat' => 11871.43,
            'residual_value_including_vat' => 11871.43 * 1.23,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('lease-vehicles.item-bulk-finance', $variant->slug), [
                'items' => [
                    'colours' => [
                        [
                            'id' => 'foo',
                            'price_ex_vat_ex_vrt' => 123.45,
                            'vrt' => 7.5,
                            // optional vat should be allowed
                            'vat' => 0.05,
                        ],
                    ],
                    'trims' => [],
                    'options' => [
                        [
                            'id' => 'bar',
                            'price_ex_vat_ex_vrt' => 456.78,
                            'vrt' => 50,
                            // null vat should be allowed
                            'vat' => null,
                        ],
                        [
                            'id' => 'baz',
                            'price_ex_vat_ex_vrt' => 456.78,
                            'vrt' => 50,
                            // vat shouldn't need to be present
                        ],
                    ],
                ],
            ]);

        $response->assertStatus(200);
        $this->assertArrayHasKey('colours', $response->json());
        $this->assertArrayHasKey('trims', $response->json());
        $this->assertArrayHasKey('options', $response->json());
        $this->assertCount(1, $response->json('colours'));
        $this->assertCount(0, $response->json('trims'));
        $this->assertCount(2, $response->json('options'));
        $this->assertEquals(2.95, $response->json('colours.foo.finance_options.monthly_prices.48-40000-12.monthly_price_ex_vat'));
        $this->assertEquals(11.42, $response->json('options.bar.finance_options.monthly_prices.48-40000-12.monthly_price_ex_vat'));
        $this->assertEquals(11.42, $response->json('options.baz.finance_options.monthly_prices.48-40000-12.monthly_price_ex_vat'));
        $this->assertEquals(3.1, $response->json('colours.foo.finance_options.monthly_prices.48-40000-12.monthly_price_inc_vat'));
        $this->assertEquals(14.04, $response->json('options.bar.finance_options.monthly_prices.48-40000-12.monthly_price_inc_vat'));
        $this->assertEquals(14.04, $response->json('options.baz.finance_options.monthly_prices.48-40000-12.monthly_price_inc_vat'));
    }

    private function createSettings()
    {
        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Enabled',
            'config_key' => 'leasing-keyloop-fleet-kompact-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Provider Type',
            'config_key' => 'leasing-keyloop-fleet-kompact-provider-type',
            'type' => 'string',
            'value' => 'kearys',
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'VAT percentage in decimal',
            'description' => 'For use in finance calculations. Decimal representation of a percentage. 7% should be entered as 0.07',
            'config_key' => 'leasing-keyloop-fleet-kompact-vat-rate',
            'type' => 'number',
            'value' => 0.23,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Interest rate percentage in decimal',
            'description' => 'For use in finance calculations. Decimal representation of a percentage. 7% should be entered as 0.07',
            'config_key' => 'leasing-keyloop-fleet-kompact-interest-rate',
            'type' => 'number',
            'value' => 0.07,
        ]);
    }
}
