<?php

namespace App\Http\Controllers;

use App\Facades\OfferFilter;
use App\Facades\Settings;
use App\Filter;
use App\Http\Requests\OfferSearchRequest;
use App\Http\Requests\ViewOfferRequest;
use App\Http\Resources\MakeListResource;
use App\Http\Resources\OfferListResource;
use App\Http\Resources\OfferResource;
use App\Http\Resources\PageResource;
use App\Traits\ModelsForMakeInSearch;
use Carbon\Carbon;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\OfferType;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;

class OfferController extends Controller
{
    use ModelsForMakeInSearch;

    public function index(OfferSearchRequest $request): array
    {
        $page = Page::query()->where('slug', 'offers')->first();
        $featured = VehicleOffer::query()->active()
            ->withListingRelationships()
            ->where('featured', 1)
            ->get();

        $makes = VehicleMake::query()
            ->with('filterIndex')
            ->whereHas('offers', fn ($query) => $query->active())
            ->get();

        $models = VehicleModel::query()
            ->whereHas('cheapestActiveOffer')
            ->with([
                'cheapestActiveOffer.primaryMediaUse.media',
                'cheapestActiveOffer.make.filterIndex',
                'cheapestActiveOffer.offerType',
                'cheapestActiveOffer.model.filterIndex',
                'make.filterIndex',
                'cheapestActiveOffer.make',
                'cheapestActiveOffer.model',
                'cheapestActiveOffer.fuelType',
                'cheapestActiveOffer.transmission',
                'cheapestActiveOffer.drivetrain',
                'cheapestActiveOffer.bodyStyle',
            ])
            ->when(
                Settings::get('offer-list-load-features'),
                fn($query) => $query->with('cheapestActiveOffer.features')
            )
            ->when($request->filled('offer_type'), fn($query) => $query
                ->whereHas(
                    'cheapestActiveOffer.offerType',
                    fn($typeQuery) => $typeQuery->where('slug', $request->input('offer_type'))
                ))
            ->get()
            ->groupBy(fn ($model) => $model->make->filterIndex->slug)
            ->map(fn ($group) => new OfferListResource($group->map(fn ($model) => $model->cheapestActiveOffer)));

        return [
            'page' => new PageResource($page),
            'featured' => new OfferListResource($featured),
            'makes' => new MakeListResource($makes),
            'models' => $models,
            'types' => OfferType::query()->orderBy('name')->get(),
        ];
    }

    public function search(OfferSearchRequest $request, string $make): array
    {
        $makeModel = $make !== 'all'
            ? Filter::matchSlugToRecord('make', $make)
            : null;
        $vehicleModel = $request->filled('model_id')
            ? Filter::matchSlugToRecord('model', $request->input('model_id'))
            : null;
        $makeId = $makeModel?->id;

        $offers = VehicleOffer::query()
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
                'offerType',
                'fuelType',
                'bodyStyle',
                'transmission',
                'cheapestFinance',
                'dealership',
            ])
            ->when($request->filled('offer_type'), fn($query) => $query
                ->whereHas('offerType', fn($typeQuery) => $typeQuery->where('slug', $request->input('offer_type'))))
            ->when($make != 'all', fn($query) => $query->where('make_id', $makeId))
            ->whereModelSlug($request->input('model_id'))
            ->setSortBy($request->input('sort_by'))
            ->active()
            ->paginate();

        return [
            'make' => $makeModel?->name ?? '',
            'model' => $vehicleModel?->name,
            'models' => $this->getModelsForMake($makeId, 'offers'),
            'offers' => new OfferListResource($offers),
        ];
    }

    public function searchWithFilters()
    {
        return OfferFilter::handle();
    }

    public function show(ViewOfferRequest $request, VehicleOffer $offer): OfferResource
    {
        $offer->load([
            'fuelType',
            'offerType',
            'transmission',
            'drivetrain',
            'bodyStyle',
            'dealership',
        ]);

        $view = $offer->views()
            ->firstOrNew(['date' => Carbon::now()->format('Y-m-d')]);
        $view->hits++;
        $view->save();
        return new OfferResource($offer);
    }
}
