<?php

namespace Tests\Feature;

use App\Facades\Feature;
use App\Facades\Settings;
use App\Models\FilterIndex;
use App\Modules\PlaceholderImages\ImaginStudio;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Models\FormQuestion;
use Mtc\Crm\Models\FormSection;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleTrim;
use Tests\TestCase;

class NewCarControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testSearch()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);

        NewCar::factory(5)
            ->create([
                'make_id' => $make->id,
                'published_at' => Carbon::now()->subDays(3),
            ]);
        NewCar::factory(3);
        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));

        $response->assertStatus(200);
        $this->assertArrayHasKey('make', $response->json());
        $this->assertEquals($make->name, $response->json('make'));
        $this->assertArrayHasKey('models', $response->json());
        $this->assertArrayHasKey('offers', $response->json());
    }

    public function testShowCar()
    {
        Http::fake([
            'https://cdn.imagin.studio/getPaints*' => Http::response(['paintData' => ['paintCombinations' => []]])
        ]);
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);
        $offer = NewCar::factory()->create(['published' => true]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));
        $response->assertStatus(200);

        $offer2 = NewCar::factory()->create(['published' => false]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer2));
        $response->assertStatus(403);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', [$offer2, 'a' => base64_encode($offer2->id . '-' . $offer2->slug)]));
        $response->assertStatus(200);
    }

    public function testShowCarWithSeoData()
    {
        $car = NewCar::factory()->create([
            'published' => true,
            'seo' => ['title' => 'foo', 'description' => 'baz bar']
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('seo.title'));
        $this->assertEquals('baz bar', $response->json('seo.description'));

        SeoDefault::query()->create([
            'section' => 'new-car',
            'title' => '{{TITLE}} | {{MAKE}} Offers | {{SITE_NAME}} ',
            'description' => '{{CONTENT_EXCERPT}}',
        ]);

        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create();
        $car = NewCar::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'published' => true,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertStringContainsString($car->name, $response->json('seo.title'));
        $this->assertStringContainsString($make->name, $response->json('seo.title'));
    }

    public function testCarHasForm()
    {
        $form = Form::factory()->create();
        $section = FormSection::factory()->create(['form_id' => $form->id]);
        FormQuestion::factory()->create([
            'form_id' => $form->id,
            'form_section_id' => $section->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);

        $car = NewCar::factory()->create(['published' => true]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);

        $car->update([
            'form_id' => $form->id,
        ]);
        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $car));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('forms.enquire'));
        $this->assertEquals($form->id, $response->json('forms.enquire.id'));
    }

    public function testCarShowsDealershipNumber()
    {
        $dealership1 = Dealership::factory()->create([
            'contact_no' => '123123',
        ]);
        $dealership2 = Dealership::factory()->create([
            'contact_no' => null
        ]);
        $offer = NewCar::factory()->create(['published' => true]);
        $offer1 = NewCar::factory()->create(['published' => true, 'dealership_id' => $dealership1]);
        $offer2 = NewCar::factory()->create(['published' => true, 'dealership_id' => $dealership2]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer1));
        $response->assertStatus(200);
        $this->assertIsArray($response->json('contact_number'));
        $this->assertArrayHasKey('number', $response->json('contact_number'));
        $this->assertArrayHasKey('display', $response->json('contact_number'));
        $this->assertStringContainsString('123123', $response->json('contact_number.number'));

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer2));
        $response->assertStatus(200);
        $this->assertNull($response->json('contact_number'));
    }

    public function testTrims()
    {
        $make = VehicleMake::factory()->create();
        $model_1 = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);
        $model_2 = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);
        $offer = NewCar::factory()->create([
            'name' => 'foo',
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'published' => true,
        ]);
        VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'name' => 'abc',
        ]);
        VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'name' => 'def',
        ]);
        VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model_2->id,
            'name' => 'xyz',
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));

        $response->assertStatus(200);
        $this->assertArrayHasKey('trims', $response->json());
        $this->assertCount(3, VehicleTrim::all());
        $this->assertCount(2, $response->json('trims'));
    }

    public function testImaginStudioUrlDisabled()
    {
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);

        $offer = NewCar::factory()->create([
            'name' => 'foo',
            'make_id' => $make->id,
            'model_id' => $model->id,
            'published' => true,
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));

        $response->assertStatus(200);
        $this->assertArrayHasKey('imagin_studio_base_url', $response->json());
        $this->assertNull($response->json('imagin_studio_base_url'));
    }

    public function testImaginStudioUrl()
    {
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);

        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);

        $offer = NewCar::factory()->create([
            'name' => 'foo',
            'make_id' => $make->id,
            'model_id' => $model->id,
            'published' => true,
        ]);

        $response = $this->asTenant(tenant())
            ->getJson(route('new-cars.show', $offer));

        $response->assertStatus(200);
        $this->assertArrayHasKey('imagin_studio_base_url', $response->json());
        $this->assertStringContainsString('cdn.imagin.studio/getImage?customer=', $response->json('imagin_studio_base_url'));
    }

    public function testFuelType()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $model = VehicleModel::factory()->create([
            'make_id' => $make->id
        ]);

        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        FilterIndex::factory()->create([
            'filter_id' => $model->id,
            'filter_type' => 'model',
        ]);

        $fuel_type_foo = FuelType::factory()->create([
            'name' => 'foo'
        ]);

        NewCar::factory(5)
            ->create([
                'make_id' => $make->id,
                'model_id' => $model->id,
                'published' => true,
                'published_at' => Carbon::now()->subDays(3),
                'fuel_type_id' => $fuel_type_foo->id,
            ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));
        $response->assertStatus(200);
        $this->assertCount(5, NewCar::all());
        $this->assertArrayHasKey('offers', $response->json());
        $this->assertArrayHasKey('fuel_type', $response->json('offers.data.0'));
        $this->assertEquals($fuel_type_foo->name, $response->json('offers.data.0.fuel_type'));
    }

    public function testImaginStudioClientKey()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'imagin-placeholders-client-key',
            'value' => 'abc123',
            'type' => 'string',
        ]);

        $new_car = NewCar::factory()->create();
        $vehicle = Vehicle::factory()->create();
        $vehicle_offer = VehicleOffer::factory()->create();

        $url = (new ImaginStudio())->getForNewCar($new_car);
        $this->assertStringContainsString('&billingTag=abc123', $url);

        $url = (new ImaginStudio())->getForVehicle($vehicle);
        $this->assertStringContainsString('&billingTag=abc123', $url);

        $url = (new ImaginStudio())->getForOffer($vehicle_offer);
        $this->assertStringContainsString('&billingTag=abc123', $url);

        $url = (new ImaginStudio())->getForOther('foo', 'baz', 'bar');
        $this->assertStringContainsString('&billingTag=abc123', $url);

        $url = (new ImaginStudio())->getBaseUrl();
        $this->assertStringContainsString('&billingTag=abc123', $url);
    }

    public function testSearchReturnsImaginStudiBaseUrl()
    {
        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);

        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();

        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);

        NewCar::factory(5)
            ->create([
                'make_id' => $make->id,
                'published_at' => Carbon::now()->subDays(3),
            ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));

        $response->assertStatus(200);

        $this->assertArrayHasKey('imagin_studio_base_url', $response->json());
        $this->assertNotEmpty($response->json('imagin_studio_base_url'));
    }

    public function testFuelTypes()
    {
        Page::factory()->create(['slug' => 'offers']);

        $make = VehicleMake::factory()->create();

        $model_1 = VehicleModel::factory()->create([
            'make_id' => $make->id
        ]);

        $model_2 = VehicleModel::factory()->create([
            'make_id' => $make->id
        ]);

        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        FilterIndex::factory()->create([
            'filter_id' => $model_1->id,
            'filter_type' => 'model',
        ]);
        FilterIndex::factory()->create([
            'filter_id' => $model_2->id,
            'filter_type' => 'model',
        ]);

        $fuel_type_foo = FuelType::factory()->create([
            'name' => 'foo'
        ]);

        NewCar::factory()
            ->create([
                'make_id' => $make->id,
                'model_id' => $model_1->id,
                'published' => true,
                'published_at' => Carbon::now()->subDays(3),
                'fuel_type_id' => $fuel_type_foo->id,
            ]);

        NewCar::factory()
            ->create([
                'make_id' => $make->id,
                'model_id' => $model_2->id,
                'published' => true,
                'published_at' => Carbon::now()->subDays(3),
                'fuel_type_id' => $fuel_type_foo->id,
            ]);

        VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'name' => 'foo',
            'fuel_types' => [
                'baz',
                'bar',
            ],
        ]);

        VehicleTrim::query()->create([
            'make_id' => $make->id,
            'model_id' => $model_2->id,
            'name' => 'foo_too',
            'fuel_types' => [
                'buzz',
                'bar',
            ],
        ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'trim' => 'foo',
        ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_2->id,
            'trim' => 'foo_too',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));
        $response->assertStatus(200);
        $this->assertCount(2, NewCar::all());
        $this->assertArrayHasKey('offers', $response->json());
        $this->assertArrayHasKey('fuel_types', $response->json('offers.data.0'));
        $this->assertArrayHasKey('bar', $response->json('offers.data.0.fuel_types'));
        $this->assertArrayHasKey('bar', $response->json('offers.data.1.fuel_types'));

        // check that items contain the expected fuel types
        if (array_key_exists('buzz', $response->json('offers.data.0.fuel_types'))) {
            $this->assertArrayHasKey('baz', $response->json('offers.data.1.fuel_types'));
        } else {
            $this->assertArrayHasKey('baz', $response->json('offers.data.0.fuel_types'));
        }
    }

    public function testFromPrice()
    {
        Page::factory()->create(['slug' => 'offers']);
        $make = VehicleMake::factory()->create();
        $model_1 = VehicleModel::factory()->create([
            'make_id' => $make->id
        ]);
        $model_2 = VehicleModel::factory()->create([
            'make_id' => $make->id
        ]);
        $index = FilterIndex::factory()->create([
            'filter_id' => $make->id,
            'filter_type' => 'make',
        ]);
        FilterIndex::factory()->create([
            'filter_id' => $model_1->id,
            'filter_type' => 'model',
        ]);
        FilterIndex::factory()->create([
            'filter_id' => $model_2->id,
            'filter_type' => 'model',
        ]);
        $fuel_type_foo = FuelType::factory()->create([
            'name' => 'foo'
        ]);
        NewCar::factory()
            ->create([
                'make_id' => $make->id,
                'model_id' => $model_1->id,
                'published' => true,
                'published_at' => Carbon::now()->subDays(3),
                'fuel_type_id' => $fuel_type_foo->id,
            ]);
        NewCar::factory()
            ->create([
                'make_id' => $make->id,
                'model_id' => $model_2->id,
                'published' => true,
                'published_at' => Carbon::now()->subDays(3),
                'fuel_type_id' => $fuel_type_foo->id,
            ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_2->id,
            'price' => 1,
        ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'price' => 789,
        ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_1->id,
            'price' => 123,
        ]);

        VehicleOffer::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_2->id,
            'price' => 456,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson(route('new-cars.search', $index->slug));
        $response->assertStatus(200);
        $this->assertCount(2, NewCar::all());
        $this->assertArrayHasKey('offers', $response->json());
        $this->assertArrayHasKey('from_price', $response->json('offers.data.0'));
        $this->assertEquals(123, $response->json('offers.data.0.from_price'));
    }
}
