<?php

namespace App\Http\Controllers;

use App\Filter;
use App\Http\Requests\OfferSearchRequest;
use App\Http\Resources\MakeListResource;
use App\Http\Resources\OfferListResource;
use App\Http\Resources\OfferResource;
use App\Http\Resources\PageResource;
use Carbon\Carbon;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;

class OfferController extends Controller
{
    public function index(OfferSearchRequest $request)
    {
        $page = Page::query()->where('slug', 'offers')->first();
        $featured = VehicleOffer::query()->active()
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
            ])
            ->where('featured', 1)
            ->get();

        $makes = VehicleMake::query()
            ->with('filterIndex')
            ->whereHas('offers', fn ($query) => $query->active())
            ->get();

        $models = VehicleModel::query()
            ->whereHas('cheapestActiveOffer')
            ->with([
                'cheapestActiveOffer.primaryMediaUse.media',
                'cheapestActiveOffer.make.filterIndex',
                'cheapestActiveOffer.model.filterIndex',
                'make.filterIndex'
            ])
            ->get()
            ->groupBy(fn ($model) => $model->make->filterIndex->slug)
            ->map(fn ($group) => new OfferListResource($group->map(fn ($model) => $model->cheapestActiveOffer)));

        return [
            'page' => new PageResource($page),
            'featured' => new OfferListResource($featured),
            'makes' => new MakeListResource($makes),
            'models' => $models,
        ];
    }

    public function search(OfferSearchRequest $request, string $make)
    {
        $makeModel = Filter::matchSlugToRecord('make', $make);
        $makeId = $makeModel->id;
        $offers = VehicleOffer::query()
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
            ])
            ->where('make_id', $makeId)
            ->when(
                $request->filled('model_id'),
                fn ($query) => $query->whereHas(
                    'model.filterIndex',
                    fn ($indexQuery) => $indexQuery->where('slug', $request->input('model_id'))
                )
            )
            ->setSortBy($request->input('sort_by'))
            ->active()
            ->paginate();

        $models = VehicleModel::query()
            ->with('filterIndex')
            ->where('make_id', $makeId)
            ->whereHas('offers', fn ($query) => $query->active())
            ->get()
            ->map(fn ($model) => [
                'code' => $model->filterIndex->slug ?? '',
                'label' => $model->name,
            ])
            ->prepend([
                'code' => null,
                'label' => 'All',
            ]);

        return [
            'make' => $makeModel->name,
            'models' => $models,
            'offers' => new OfferListResource($offers),
        ];
    }

    /**
     * Show vehicle FPA details
     *
     * @param Vehicle $vehicle
     * @return OfferResource
     */
    public function show(VehicleOffer $offer)
    {
        $view = $offer->views()
            ->firstOrNew(['date' => Carbon::now()->format('Y-m-d')]);
        $view->hits++;
        $view->save();
        return new OfferResource($offer);
    }
}
