<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\IntegrationRepository;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Form;
use Mtc\MercuryDataModels\SeoDefault;

class VehicleResource extends JsonResource
{
    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->fuel_type = $this->resource->fuelType?->name;
        $this->resource->transmission = $this->resource->transmission?->name;
        $this->resource->body_style = $this->resource->bodyStyle?->name;
        $this->resource->load([
            'mediaUses.media',
            'specs',
        ]);
        $this->resource->media_uses = $this->resource->mediaUses->map(fn($mediaUse) => [
            'title' => $mediaUse->title,
            'alt_text' => $mediaUse->alt_text,
            'description' => $mediaUse->description,
            'caption' => $mediaUse->caption,
            'src_large' => $mediaUse->getUrl('900x600'),
            'src_medium' => $mediaUse->getUrl('440x295'),
            'type' => $mediaUse->media->type
        ]);
        $this->resource->specs = $this->resource->specs->map(fn($spec) => [
            'description' => $spec->description,
        ]);

        $this->resource->dealership = $this->resource->dealership?->only([
            'slug',
            'name',
            'contact_no',
            'open_time_text',
        ]);

        $this->resource->actions = [
            'deposit' => $this->depositAction(),
            'deal' => $this->salesAction(),
            'finance' => $this->financeAction(),
            'enquiry' => $this->enquireAction(),
            'valuation' => $this->valuationAction(),
        ];

        $this->resource->finance = $this->resource->financeExamples()
            ->orderBy('monthly_price')
            ->first();

        $this->resource->content = $this->getContent();

        $resource = $this->resource->only([
            'title',
            'odometer_mi',
            'odometer_km',
            'price',
            'deposit',
            'original_price',
            'monthly_price',
            'transmission',
            'fuel_type',
            'registration_number',
            'engine_size_cc',
            'seats',
            'colour',
            'media_uses',
            'specs',
            'content',
            'actions',
            'finance',
            'dealership',
        ]);

        $resource['seo'] = $this->fillSeo();

        return $resource;
    }

    private function salesAction()
    {
        $salesHandlers = (new IntegrationRepository())->getEnabledForType('sales');
        if ($salesHandlers->isEmpty()) {
            return false;
        }

        return App::make($salesHandlers->first()['class'])->getVehicleAction($this->resource);
    }

    private function getContent(): Collection
    {
        return collect([
            [
                'title' => '',
                'content' => $this->resource->description,
                'buttonLink' => false
            ], [

                'title' => 'Quality Promise',
                'content' => 'We guarantee that every Autonomy car has been prepped to a high standard and has passed '
                    . 'a 114 point mechanical check to ensure the condition and safety of '
                    . 'the vehicle is uncompromised.',
                'buttonText' => 'Find out more',
                'buttonLink' => '/quality-promise',
            ],
        ])->filter(fn ($entry) => !empty($entry['content']));
    }

    private function fillSeo(): array
    {
        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }

        $default = SeoDefault::query()->where('section', 'vehicle')->first();
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{SITE_NAME}}' => Settings::get('app-name'),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' => $this->resource->title,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{CONTENT_EXCERPT}}' => Str::limit($this->resource->description, 150),

            ], $default->description ?? '');
        }
        return $seoData;
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }

    private function valuationAction()
    {
        return true;
    }

    private function enquireAction(): ?array
    {
        $form = Form::query()
            ->with('sections.questions')
            ->whereHas('type', fn($query) => $query->where('name', 'Vehicle Enquiry'))
            ->first();
        return [
            'type' => 'form',
            'title' => 'Make an enquiry',
            'details' => $form
                ? (new FormViewResource($form))->setValues([ 'vehicle_id' => $this->resource->id ])
                : [],
        ];
    }

    private function financeAction(): bool|array
    {
        return false;
    }

    private function depositAction(): bool|array
    {
        return false;
    }
}
