<?php

namespace App;

use App\Facades\Settings;
use App\Sales\SilverBullet;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Mtc\MercuryDataModels\Setting;

class IntegrationRepository
{
    protected static $concurrentEnabledCountLimit = [
        'sales-channels' => [
            'lite' => 2,
            'standard' => 5,
            'pro' => 999,
            'enterprise' => 999,
        ],
    ];

    protected static $integrations = [
        'silver-bullet' => [
            'type' => 'sales',
            'config_key' => 'sales-silver-bullet-enabled',
            'link' => '/',
            'class' => SilverBullet::class,
            'image' => [
                'src' => 'silverbullet.svg',
                'alt' => 'SilverBullet',
            ],
        ],
        'ivendi' => [
            'type' => 'finance',
            'config_key' => 'finance-ivendi-enabled',
            'link' => '/',
            'image' => [
                'src' => 'ivendi.png',
                'alt' => 'iVendi',
            ],
        ],
        'code-weavers' => [
            'type' => 'finance',
            'config_key' => 'finance-codeweavers-enabled',
            'link' => '/',
            'image' => [
                'src' => 'codeweavers.png',
                'alt' => 'CodeWeavers',
            ],
        ],
        'evolution-funding' => [
            'type' => 'finance',
            'config_key' => 'finance-evolution-enabled',
            'link' => '/',
            'image' => [
                'src' => 'evolution-funding.png',
                'alt' => 'Evolution Funding',
            ],
        ],
        'enquiry-max' => [
            'type' => 'crm',
            'config_key' => 'crm-enquiry-max-enabled',
            'link' => '/',
            'image' => [
                'src' => 'enquiry-max.png',
                'alt' => 'EnquiryMax',
            ],
        ],
        'rapid-rtc' => [
            'type' => 'crm',
            'config_key' => 'crm-rapid-rtc-enabled',
            'link' => '/',
            'image' => [
                'src' => 'rapidrtc.svg',
                'alt' => 'Rapid RTC CRM',
            ],
        ],
        'mail-chimp' => [
            'type' => 'newsletters',
            'config_key' => 'newsletters-mailchimp-enabled',
            'link' => '/',
            'image' => [
                'src' => 'mailchimp.png',
                'alt' => 'MailChimp',
            ],
        ],
        'campaign-monitor' => [
            'type' => 'newsletters',
            'config_key' => 'newsletters-campaign-monitor-enabled',
            'link' => '/',
            'image' => [
                'src' => 'campaign-monitor.png',
                'alt' => 'Campaign Monitor',
            ],
        ],
        'cap' => [
            'type' => 'vehicle-specs',
            'config_key' => 'vehicle-spec-providers-cap-enabled',
            'link' => '/',
            'image' => [
                'src' => 'cap.png',
                'alt' => ' CAP Data',
            ],
        ],
        'autos-on-show' => [
            'type' => 'image-sync',
            'config_key' => 'image-sync-autos-on-show-enabled',
            'link' => '/',
            'image' => [
                'src' => 'autos-on-show.png',
                'alt' => 'Autos On Show',
            ],
        ],
        'auto-imaging' => [
            'type' => 'image-sync',
            'config_key' => 'image-sync-auto-imaging-enabled',
            'link' => '/',
            'image' => [
                'src' => 'auto-imaging.png',
                'alt' => 'AutoImaging',
            ],
        ],
        'zendesk' => [
            'type' => 'chat',
            'config_key' => 'chat-zendesk-enabled',
            'link' => '/',
            'widget' => 'ZenDeskChat',
            'data' => [
                'key' => 'chat-zendesk-key',
            ],
            'image' => [
                'src' => 'zendesk.png',
                'alt' => 'ZenDesk',
            ],
        ],
        'rapid-rtc-chat' => [
            'type' => 'chat',
            'config_key' => 'chat-rapid-rtc-enabled',
            'link' => '/',
            'widget' => 'RapidRtcChat',
            'data' => [
                'key' => 'chat-rapid-rtc-key',
            ],
            'image' => [
                'src' => 'rapidrtc.svg',
                'alt' => 'Rapid RTC Live Chat',
            ],
        ],
        'gubagoo' => [
            'type' => 'chat',
            'config_key' => 'chat-gubagoo-enabled',
            'link' => '/',
            'widget' => 'GubagooChat',
            'data' => [
                'key' => 'chat-gubagoo-account_id',
            ],
            'image' => [
                'src' => 'gubagoo.png',
                'alt' => 'Gubagoo',
            ],
        ],
        'live-chat' => [
            'type' => 'chat',
            'config_key' => 'chat-live-chat-enabled',
            'link' => '/',
            'widget' => 'LiveChat',
            'data' => [
                'client_id' => 'chat-live-chat-client_id',
                'license_id' => 'chat-live-chat-license_id',
            ],
            'image' => [
                'src' => 'live-chat.png',
                'alt' => ' Live Chat',
            ],
        ],
        'live-person' => [
            'type' => 'chat',
            'config_key' => 'chat-live-person-enabled',
            'link' => '/',
            'widget' => 'LivePerson',
            'data' => [
                'key' => 'chat-live-person-client_id',
            ],
            'image' => [
                'src' => 'live-chat.png',
                'alt' => ' Live Chat',
            ],
        ],
        'car-gurus' => [
            'type' => 'sales-channels',
            'config_key' => 'sales-channels-car-gurus-enabled',
            'link' => '/',
            'image' => [
                'src' => 'car-gurus.svg',
                'alt' => ' Car Gurus Feed',
            ],
        ],
        'motors-co-uk' => [
            'type' => 'sales-channels',
            'config_key' => 'sales-channels-motors-co-uk-enabled',
            'link' => '/',
            'image' => [
                'src' => 'motors-co-uk.png',
                'alt' => 'Motors.co.uk Feed',
            ],
        ],
        'pinewood' => [
            'type' => 'stock',
            'config_key' => 'stock-pinewood-enabled',
            'link' => '/',
            'image' => [
                'src' => 'pinewood.png',
                'alt' => 'Pinnacle Pinewood',
            ],
        ],
        'dms-navigator' => [
            'type' => 'stock',
            'config_key' => 'stock-dms-navigator-enabled',
            'link' => '/',
            'image' => [
                'src' => 'navigator.png',
                'alt' => 'DMS Navigator',
            ],
        ],
        'facebook' => [
            'type' => 'socials',
            'config_key' => 'app-socials-facebook',
        ],
        'twitter' => [
            'type' => 'socials',
            'config_key' => 'app-socials-twitter',
        ],
        'youtube' => [
            'type' => 'socials',
            'config_key' => 'app-socials-youtube',
        ],
        'instagram' => [
            'type' => 'socials',
            'config_key' => 'app-socials-instagram',
        ],
        'linkedin' => [
            'type' => 'socials',
            'config_key' => 'app-socials-linkedin',
        ],
    ];

    /**
     * Get list of enabled integrations
     *
     * @return array
     */
    public function getEnabled(): array
    {
        $enabled = $this->findEnabledIntegrations();
        return collect(self::$integrations)
            ->filter(fn($integration) => isset($enabled[$integration['config_key']]))
            ->map(function ($integration) {
                unset($integration['config_key']);
                $integration['image']['src'] = Storage::disk('file-storage')
                    ->url('assets/logos/' . $integration['image']['src']);
                return $integration;
            })
            ->values()
            ->toArray();
    }

    /**
     * Find all integrations for a specific type
     *
     * @param string $type
     * @return Collection
     */
    public function getAllForType(string $type, bool $checkEnabled = false): Collection
    {
        return collect(self::$integrations)
            ->filter(fn ($integration) => $integration['type'] === $type)
            ->map(function ($integration) use ($checkEnabled) {
                if ($checkEnabled) {
                    $integration['active'] =  Settings::get($integration['config_key']);
                }
                return $integration;
            });
    }

    /**
     * Find enabled integrations for a specific type
     *
     * @param string $type
     * @return \Illuminate\Support\Collection
     */
    public function getEnabledForType(string $type)
    {
        $enabled = $this->findEnabledIntegrations();
        return collect(self::$integrations)
            ->filter(fn ($integration) => $integration['type'] === $type)
            ->filter(fn($integration) => isset($enabled[$integration['config_key']]));
    }

    /**
     * Find entries of type that have set values
     *
     * @param string $type
     * @return Collection
     */
    public function findNonEmptyValuesForType(string $type)
    {
        return $this->getAllForType($type, true)
            ->filter(fn ($integration) => !empty($integration['active']))
            ->map(fn ($integration) => $integration['active']);
    }

    /**
     * Check if given config key is enable flag for an integration
     *
     * @param string $key
     * @return bool
     */
    public function keyIsEnableSettingValue(string $key): bool
    {
        return collect(self::$integrations)
            ->filter(fn($integration) => $integration['config_key'] === $key)
            ->isNotEmpty();
    }

    /**
     * Get integration type from enable key
     *
     * @param string $key
     * @return string|null
     */
    public function getTypeFromKey(string $key): ?string
    {
        return collect(self::$integrations)
                ->filter(fn($integration) => $integration['config_key'] === $key)
                ->first()['type'] ?? null;
    }


    /**
     * Check how many integrations of type are enabled in input
     *
     * @param array $input
     * @param string $type
     * @return int
     */
    public function checkEnabledCountForType(array $input, string $type): int
    {
        $type_enable_flags = collect(self::$integrations)
            ->filter(fn($integration) => $integration['type'] === $type)
            ->pluck('config_key');

        return collect($input)
            ->filter(fn($value, $key) => $type_enable_flags->search($key) !== false && $value)
            ->count();
    }

    /**
     * Find enabled integrations from settings
     *
     * @return Collection
     */
    protected function findEnabledIntegrations(): Collection
    {
        return Setting::query()
            ->whereIn('config_key', collect(self::$integrations)->pluck('config_key'))
            ->where('value', 1)
            ->get()
            ->keyBy('config_key');
    }
}
