<?php

namespace App;

use App\Contracts\ProvidesIntegrationToken;
use App\Facades\Settings;
use App\Http\Resources\LiveChatFactory;
use App\Integrations\AutosOnShow;
use App\Sales\SilverBullet;
use App\Tracking\AnalyticsTracker;
use App\Tracking\CalltracksTracking;
use App\Tracking\FacebookTracker;
use App\Tracking\HotJarTracking;
use App\Tracking\LinkedInTracker;
use App\Tracking\MicrosoftClarityTracker;
use App\Tracking\PinterestTracker;
use App\Tracking\RedditTracker;
use App\Tracking\SnapchatTracker;
use App\Tracking\TikTokTracker;
use App\Tracking\UserBackTracking;
use App\Tracking\XTracker;
use Illuminate\Support\Collection;

class IntegrationRepository
{
    protected static $concurrentEnabledCountLimit = [
        'sales-channels' => [
            'lite' => 2,
            'standard' => 5,
            'pro' => 999,
            'enterprise' => 999,
        ],
    ];

    protected static $integrations = [
        'silver-bullet' => [
            'type' => 'sales',
            'config_key' => 'sales-silver-bullet-enabled',
            'link' => '/',
            'class' => SilverBullet::class,
            'image' => [
                'src' => 'silverbullet.svg',
                'alt' => 'SilverBullet',
            ],
        ],
        'ivendi' => [
            'type' => 'finance',
            'config_key' => 'finance-ivendi-enabled',
            'link' => '/',
            'image' => [
                'src' => 'ivendi.png',
                'alt' => 'iVendi',
            ],
        ],
        'code-weavers' => [
            'type' => 'finance',
            'config_key' => 'finance-codeweavers-enabled',
            'link' => '/',
            'image' => [
                'src' => 'codeweavers.png',
                'alt' => 'CodeWeavers',
            ],
        ],
        'evolution-funding' => [
            'type' => 'finance',
            'config_key' => 'finance-evolution-enabled',
            'link' => '/',
            'image' => [
                'src' => 'evolution-funding.png',
                'alt' => 'Evolution Funding',
            ],
        ],
        'enquiry-max' => [
            'type' => 'crm',
            'config_key' => 'crm-enquiry-max-enabled',
            'link' => '/',
            'image' => [
                'src' => 'enquiry-max.png',
                'alt' => 'EnquiryMax',
            ],
        ],
        'rapid-rtc' => [
            'type' => 'crm',
            'config_key' => 'crm-rapid-rtc-enabled',
            'link' => '/',
            'image' => [
                'src' => 'rapidrtc.svg',
                'alt' => 'Rapid RTC CRM',
            ],
        ],
        'mail-chimp' => [
            'type' => 'newsletters',
            'config_key' => 'newsletters-mailchimp-enabled',
            'link' => '/',
            'image' => [
                'src' => 'mailchimp.png',
                'alt' => 'MailChimp',
            ],
        ],
        'campaign-monitor' => [
            'type' => 'newsletters',
            'config_key' => 'newsletters-campaign-monitor-enabled',
            'link' => '/',
            'image' => [
                'src' => 'campaign-monitor.png',
                'alt' => 'Campaign Monitor',
            ],
        ],
        'cap' => [
            'type' => 'vehicle-specs',
            'config_key' => 'vehicle-spec-providers-cap-enabled',
            'link' => '/',
            'image' => [
                'src' => 'cap.png',
                'alt' => ' CAP Data',
            ],
        ],
        'autos-on-show' => [
            'type' => 'image-sync',
            'config_key' => 'image-sync-autos-on-show-enabled',
            'link' => '/',
            'class' => AutosOnShow::class,
            'image' => [
                'src' => 'autos-on-show.png',
                'alt' => 'Autos On Show',
            ],
        ],
        'auto-imaging' => [
            'type' => 'image-sync',
            'config_key' => 'image-sync-auto-imaging-enabled',
            'link' => '/',
            'image' => [
                'src' => 'auto-imaging.png',
                'alt' => 'AutoImaging',
            ],
        ],
        'zendesk' => [
            'type' => 'chat',
            'config_key' => 'chat-zendesk-enabled',
            'link' => '/',
            'widget' => 'ZenDeskChat',
            'data' => [
                'key' => 'chat-zendesk-key',
            ],
            'image' => [
                'src' => 'zendesk.png',
                'alt' => 'ZenDesk',
            ],
        ],
        'fuzey' => [
            'type' => 'chat',
            'config_key' => 'chat-fuzey-enabled',
            'link' => '/',
            'widget' => 'FuzeyChat',
            'data' => [
                'key' => 'chat-fuzey-client_id',
            ],
            'image' => [
                'src' => 'fuzey.webp',
                'alt' => 'Fuzey',
            ],
        ],
        'activ-engage' => [
            'type' => 'chat',
            'config_key' => 'chat-activ-engage-enabled',
            'link' => '/',
            'widget' => 'ActivEngageChat',
            'data' => [],
            'image' => [
                'src' => 'activengage.svg',
                'alt' => 'ActivEngage',
            ],
        ],
        'calltracks' => [
            'type' => 'tracking',
            'config_key' => 'crm-calltracks-enabled',
            'class' => CalltracksTracking::class,
        ],
        'easichat' => [
            'type' => 'chat',
            'config_key' => 'easi-chat-enabled',
            'link' => '/',
            'widget' => 'EasiChat',
            'data' => [
                'key' => 'easi-chat-client_id',
            ],
            'image' => [
                'src' => 'easichat.png',
                'alt' => 'EasiChat',
            ],
        ],
        'visitor' => [
            'type' => 'chat',
            'config_key' => 'chat-visitor-enabled',
            'link' => '/',
            'widget' => 'VisitorChat',
            'data' => [
                'sbText' => 'chat-visitor-sb_text',
                'ctaTitle' => 'chat-visitor-cta_title',
                'ctaText' => 'chat-visitor-cta_text',
                'id' => 'chat-visitor-id',
                'name' => 'chat-visitor-name',
                'accountId' => 'chat-account_id-name',
            ],
            'image' => [
                'src' => 'visitorchat.png',
                'alt' => 'VisitorChat',
            ],
        ],
        'rapid-rtc-chat' => [
            'type' => 'chat',
            'config_key' => 'chat-rapid-rtc-enabled',
            'link' => '/',
            'widget' => 'RapidRtcChat',
            'data' => [
                'key' => 'chat-rapid-rtc-key',
            ],
            'image' => [
                'src' => 'rapidrtc.svg',
                'alt' => 'Rapid RTC Live Chat',
            ],
        ],
        'gubagoo' => [
            'type' => 'chat',
            'config_key' => 'chat-gubagoo-enabled',
            'link' => '/',
            'widget' => 'GubagooChat',
            'data' => [
                'key' => 'chat-gubagoo-account_id',
            ],
            'image' => [
                'src' => 'gubagoo.png',
                'alt' => 'Gubagoo',
            ],
        ],
        'live-chat' => [
            'type' => 'chat',
            'config_key' => 'chat-live-chat-enabled',
            'link' => '/',
            'widget' => 'LiveChat',
            'data' => [
                'client_id' => 'chat-live-chat-client_id',
                'license_id' => 'chat-live-chat-license_id',
            ],
            'image' => [
                'src' => 'live-chat.png',
                'alt' => ' Live Chat',
            ],
        ],
        'live-person' => [
            'type' => 'chat',
            'config_key' => 'chat-live-person-enabled',
            'link' => '/',
            'widget' => 'LivePerson',
            'data' => [
                'key' => 'chat-live-person-client_id',
            ],
            'image' => [
                'src' => 'live-chat.png',
                'alt' => ' Live Chat',
            ],
        ],
        'moneypenny' => [
            'type' => 'chat',
            'config_key' => 'chat-moneypenny-enabled',
            'link' => '/',
            'widget' => 'Moneypenny',
            'data' => [
                'key' => 'chat-moneypenny-key',
            ],
        ],
        'autoconverse' => [
            'type' => 'chat',
            'config_key' => 'chat-autoconverse-enabled',
            'link' => '/',
            'widget' => 'AutoConverse',
            'data' => [
                'key' => 'chat-autoconverse-key',
            ],
        ],
        'calldrip' => [
            'type' => 'chat',
            'config_key' => 'chat-calldrip-enabled',
            'link' => '/',
            'widget' => 'CallDrip',
            'data' => [
                'key' => 'chat-calldrip-key',
                'id' => 'chat-calldrip-id',
            ],
        ],
        'car-gurus' => [
            'type' => 'sales-channels',
            'config_key' => 'sales-channels-car-gurus-enabled',
            'link' => '/',
            'image' => [
                'src' => 'car-gurus.svg',
                'alt' => ' Car Gurus Feed',
            ],
        ],
        'motors-co-uk' => [
            'type' => 'sales-channels',
            'config_key' => 'sales-channels-motors-co-uk-enabled',
            'link' => '/',
            'image' => [
                'src' => 'motors-co-uk.png',
                'alt' => 'Motors.co.uk Feed',
            ],
        ],
        'pinewood' => [
            'type' => 'stock',
            'config_key' => 'stock-pinewood-enabled',
            'link' => '/',
            'image' => [
                'src' => 'pinewood.png',
                'alt' => 'Pinnacle Pinewood',
            ],
        ],
        'dms-navigator' => [
            'type' => 'stock',
            'config_key' => 'stock-dms-navigator-enabled',
            'link' => '/',
            'image' => [
                'src' => 'navigator.png',
                'alt' => 'DMS Navigator',
            ],
        ],
        'tiktok' => [
            'type' => 'socials',
            'config_key' => 'app-socials-tiktok',
        ],
        'facebook' => [
            'type' => 'socials',
            'config_key' => 'app-socials-facebook',
        ],
        'twitter' => [
            'type' => 'socials',
            'config_key' => 'app-socials-twitter',
        ],
        'youtube' => [
            'type' => 'socials',
            'config_key' => 'app-socials-youtube',
        ],
        'instagram' => [
            'type' => 'socials',
            'config_key' => 'app-socials-instagram',
        ],
        'linkedin' => [
            'type' => 'socials',
            'config_key' => 'app-socials-linkedin',
        ],
        'analytics' => [
            'type' => 'tracking',
            'config_key' => 'seo-ga_id',
            'class' => AnalyticsTracker::class,
        ],
        'tik-tok' => [
            'type' => 'tracking',
            'config_key' => 'tracking-tiktok-code',
            'class' => TikTokTracker::class,
        ],
        'linked-in' => [
            'type' => 'tracking',
            'config_key' => 'tracking-linkedin-code',
            'class' => LinkedInTracker::class,
        ],
        'facebook-pixel' => [
            'type' => 'tracking',
            'config_key' => 'tracking-facebook-code',
            'class' => FacebookTracker::class,
        ],
        'hotjar' => [
            'type' => 'tracking',
            'config_key' => 'tracking-hotjar-code',
            'class' => HotJarTracking::class,
        ],
        'userback' => [
            'type' => 'tracking',
            'config_key' => 'tracking-userback-token',
            'class' => UserBackTracking::class,
        ],
        'reddit' => [
            'type' => 'tracking',
            'config_key' => 'tracking-reddit-pixel-id',
            'class' => RedditTracker::class,
        ],
        'snapchat' => [
            'type' => 'tracking',
            'config_key' => 'tracking-snapchat-pixel-id',
            'class' => SnapchatTracker::class,
        ],
        'x' => [
            'type' => 'tracking',
            'config_key' => 'tracking-x-pixel-id',
            'class' => XTracker::class,
        ],
        'pinterest' => [
            'type' => 'tracking',
            'config_key' => 'tracking-pinterest-tag-id',
            'class' => PinterestTracker::class,
        ],
        'microsoft_clarity' => [
            'type' => 'tracking',
            'config_key' => 'tracking-microsoft-clarity-project-id',
            'class' => MicrosoftClarityTracker::class,
        ],
    ];

    /**
     * Find all integrations for a specific type
     *
     * @param string $type
     * @return Collection
     */
    public function getAllForType(string $type, bool $checkEnabled = false): Collection
    {
        return collect(self::$integrations)
            ->filter(fn ($integration) => $integration['type'] === $type)
            ->map(function ($integration) use ($checkEnabled) {
                if ($checkEnabled) {
                    $integration['active'] =  Settings::get($integration['config_key']);
                }
                return $integration;
            });
    }

    /**
     * Find enabled integrations for a specific type
     *
     * @param string $type
     * @return \Illuminate\Support\Collection
     */
    public function getEnabledForType(string $type)
    {
        return $this->getAllForType($type, true)
            ->filter(fn($entry) => !empty($entry['active']));
    }

    /**
     * Find entries of type that have set values
     *
     * @param string $type
     * @return Collection
     */
    public function findNonEmptyValuesForType(string $type)
    {
        return $this->getAllForType($type, true)
            ->filter(fn ($integration) => !empty($integration['active']))
            ->map(fn ($integration) => $integration['active']);
    }

    public function getLiveChat()
    {
        $live_chat = $this->getEnabledForType('chat')->map(function ($data, $key) {
            $data['name'] = $key;
            return $data;
        })->first();

        return (new LiveChatFactory())->widget($live_chat);
    }

    public function getEnabledIntegration(string $name): ?array
    {
        $integration = $this->getIntegration($name);

        if (!$integration) {
            return null;
        }

        return (Settings::get($integration['config_key']) ?? false) ? $integration : null;
    }

    public function getIntegration(string $name): ?array
    {
        return self::$integrations[$name] ?? null;
    }

    public function supportsToken(array|null $integration): bool
    {
        if (!$integration || !isset($integration['class'])) {
            return false;
        }

        $class = $integration['class'];

        return class_exists($class) && is_subclass_of($class, ProvidesIntegrationToken::class);
    }
}
