<?php

namespace App;

use App\Facades\Settings;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\FilterObject;
use Mtc\Filter\Contracts\FilterSeoContract;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Filter\FilterIndex;

class Filter extends \Mtc\Filter\Filter
{
    /**
     * Filter constructor
     *
     * @param Request $request
     * @param FilterObject $product_handler
     * @param FilterSeoContract $seo
     */
    public function __construct(Request $request, FilterObject $product_handler, FilterSeoContract $seo)
    {
        $this->request = $request;
        $this->seo = $seo;
        $this->config = Config::get('filter');
        $this->product_handler = $product_handler;
        $this->active_sort_option_name = $this->config['default_sort_choice'];
    }

    /**
     * Find the id of the slug
     *
     * @param string $model_type
     * @param $slug
     * @return int
     */
    public static function matchSlugToId(string $model_type, $slug)
    {
        return FilterIndex::query()
            ->where('filter_type', $model_type)
            ->where('slug', $slug)
            ->firstOrFail()
            ->filter_id;
    }

    /**
     * Parse Request to selections and filter data
     *
     * @return array
     */
    public function parseRequest(): array
    {
        $this->decodeRequest();

        return [
            'seo' => $this->getPageSeoData(),
            'selections' => $this->getSelectionList(),
            'sort_by' => $this->active_sort_option_name,
            'base_url' => url($this->config['url_entrypoint']),
            'page' => $this->request->input('page', 1),
        ];
    }

    /**
     * Handle Ajax request, perform filtering
     *
     * @return array
     */
    public function handle(): array
    {
        $this->request = request();
        if ($this->request->has('slug')) {
            $this->filter_url_elements = explode('/', $this->request->input('slug'));
            $this->matchSortFromUrlElements();
            $this->matchSelectedFilters();
            $this->matchCustomSelections();
            $this->checkForSearchTerm();
        } else {
            $this->selections = $this->groupSelectionsByType($this->request->input('selections') ?? []);
        }
        $this->run();

        return [
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'sort_options' => $this->getSortOptions(),
            'sort_by' => $this->active_sort_option_name,
            'selections' => $this->getSelectionList(),
            'url' => $this->getPageUrl(),
            'seo' => $this->getPageSeoData(),
        ];
    }

    /**
     * Get enabled filters
     *
     * @return Collection
     */
    public function getFilters(): Collection
    {
        if (empty($this->filters)) {
            $this->filters = collect($this->config['filters'])
                ->map(fn($filter_class) => App::make($filter_class))
                ->filter(fn($filter) => $filter instanceof IsFilter)
                ->filter(fn($filter, $name) => Settings::get("automotive-vehicle-filters-$name") === true);
        }

        return $this->filters;
    }

    /**
     * Fetch the list of supported sort options
     * Returned in format of slug => display name
     *
     * @return array
     */
    public function getSortOptions(): array
    {
        return collect($this->config['sort_options'])
            ->filter(fn($filter, $filter_name) => Settings::get("automotive-vehicle-sorting-$filter_name") === true)
            ->map(fn($sort_class, $sort_name) => [
                'code' => $sort_name,
                'label' => __("filter::filter.sort_options.$sort_name")
            ])
            ->values()
            ->toArray();
    }

    /**
     * Get the selections as a flat list
     *
     * @return Collection
     */
    public function getSelectionList(): Collection
    {
        if ($this->index === null) {
            $this->getFilterIndexForSelections();
        }

        return collect($this->selections)
            ->flatMap(fn($filter_group, $filter_type) => collect($filter_group)
                ->map(fn($value) => [
                    'type' => $filter_type,
                    'value' => $value,
                    'name' => $this->selectionNameInIndex($filter_type, $value)
                ]));
    }

    /**
     * Build the URL of the page that has been viewed
     *
     * @param bool $absolute
     * @return string
     */
    public function getPageUrl(bool $absolute = true): string
    {
        $append = '';
        $customSort = $this->active_sort_option_name !== $this->config['default_sort_choice'];
        $params = array_filter([
            'page' => $this->request->input('page') > 1 ? $this->request->input('page') : null,
            'sort_by' => $customSort ? $this->active_sort_option_name : null,
        ]);
        if (!empty($params)) {
            $append = '?' . http_build_query($params);
        }
        return $this->filterElementsToSlugs($this->selections)->join('/') . $append;
    }


    /**
     * Get the selection name from index
     *
     * @param string $type
     * @param $value
     * @return string|null
     */
    protected function selectionNameInIndex(string $type, $value): ?string
    {
        return $this->index
            ->where('filter_type', $type)
            ->where('filter_id', $value)
            ->first()
            ?->name;
    }

    /**
     * Create a list of all filter elements (selections, ordering, terms etc.) in one list of slugs
     *
     * @param array $selections
     * @return Collection
     */
    protected function filterElementsToSlugs(array $selections): Collection
    {
        $all = collect([]);
        collect($selections)
            ->sortBy(fn($selection, $type) => array_search($type, array_keys($this->config['filters']), true))
            ->each(function ($type_selections, $type) use ($all) {
                $filter = $this->getFilter($type);
                // Unsupported selection
                if ($filter === null) {
                    return;
                }

                collect($type_selections)->each(function ($selection) use ($all, $filter, $type) {
                    $slugs = $this->getFilterIndexForSelections()
                        ->groupBy('filter_type')
                        ->map(fn(Collection $group) => $group->keyBy('filter_id'));

                    if ($filter instanceof CustomPatternFilter) {
                        $all->push($filter->createSlug($selection));
                    } elseif (isset($slugs[$type][$selection]['slug'])) {
                        $all->push($slugs[$type][$selection]['slug']);
                    }
                });
            });

        return $all;
    }

    /**
     * Retrieve filter results based on selections
     *
     * @return array
     */
    protected function getFilterResults(): array
    {
        return $this->getFilters()
            ->map(fn(IsFilter $filter, $name) => $filter->format($this->retrieveSingleFilterResults($filter, $name)))
            ->filter()
            ->map(function (array $filterGroup, $type) {
                $filterGroup['results'] = collect($filterGroup['results'])->map(function ($entry) use ($type) {
                    if (is_array($entry)) {
                        $entry['selected'] = isset($this->selections[$type])
                            && in_array($entry['id'], $this->selections[$type]);
                    }
                    return $entry;
                });
                return $filterGroup;
            })
            ->toArray();
    }
}
