<?php

namespace App\Seo;

use App\Facades\Settings;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\Filter\Contracts\SeoMatchingMechanism;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\VehicleMake;

class MatchSeoDefaults implements SeoMatchingMechanism
{
    /**
     * Perform a check if this mechanism should be used for setting up SEO
     * data for the page
     *
     * @param FilterInstance $filter
     * @return bool
     */
    public function matchesRequest(FilterInstance $filter): bool
    {
        return SeoDefault::query()
            ->where('section', 'filter')
            ->exists();
    }

    /**
     * Obtain SEO data for given filter object/page
     *
     * @param FilterInstance $filter
     * @return array
     */
    public function getData(FilterInstance $filter): array
    {
        $pattern = SeoDefault::query()
            ->where('section', 'filter')
            ->first();

        return [
            'title' => $this->processText($pattern->title ?? '', $filter),
            'description' => $this->processText($pattern->description ?? '', $filter),
        ];
    }

    /**
     * Process replacement tags
     *
     * @param string $input
     * @param FilterInstance $filter
     * @return string
     */
    protected function processText(string $input, FilterInstance $filter): string
    {
        $tags = $this->replacementTags($filter);
        $replaced = str_replace(array_keys($tags), $tags, $input);

        //remove unmatched keywords and resulting whitespace surplus
        $replaced = preg_replace('/{[^}]+}?/', '', $replaced);
        $replaced = preg_replace('/(\s+)/', ' ', $replaced);
        return preg_replace('/ (- )+/', ' - ', $replaced);
    }

    /**
     * Match tags with values
     *
     * @param FilterInstance $filter
     * @return array
     */
    protected function replacementTags(FilterInstance $filter)
    {
        return [
            '{{SITE_NAME}}' => Settings::get('app-name'),
            '{{PAGE}}' => __('pagination.page_x', ['page' => $filter->getCurrentPage()]),
            '{{MAKE}}' => $this->getBrandName($filter->getSelections('make')),
        ];
    }

    /**
     * Find brand name
     *
     * @param array $selections
     * @return string
     */
    protected function getBrandName(array $selections): string
    {
        if (empty($selections)) {
            return '';
        }

        return VehicleMake::query()->find(array_shift($selections))->name ?? '';
    }
}
