<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Mtc\Checkout\Events\PaidInvoice;
use Mtc\Checkout\Invoice;
use Mtc\Checkout\InvoiceRepository;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;
use Mtc\VehicleReservations\Listeners\ConfirmReservationOnInvoicePayment;
use Mtc\VehicleReservations\Reservation;
use Mtc\VehicleReservations\ReservationRepository;
use Tests\TestCase;

class KeyloopApiTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testReservationWithoutIntegration()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => false,
            'is_sold' => false,
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(201);
    }

    public function testReservationWithFailingIntegration()
    {
        // fake a 401 response for availability lookups
        Http::fake([
            "https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials"
                => Http::response($this->tokenResponse()),
            "https://api.eu.keyloop.io/buzz/*"
            => Http::response([], 401),
            "*" => Http::response([], 400),
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-id',
            'value' => 'test-client-id',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-secret',
            'value' => 'test-client-secret',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enterprise-id',
            'value' => 'buzz',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => false,
            'is_sold' => false,
        ]);

        $vehicle_attribute = VehicleAttribute::factory()->create([
            'slug' => 'keyloop-id',
        ]);

        VehicleAttributeValue::factory()->create([
            'owner_type' => 'vehicle',
            'owner_id' => $vehicle->id,
            'slug' => $vehicle_attribute->slug,
            'value' => 'bar',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(201);

        // assert that the reservation was created, despite no access to availability API
        $this->assertCount(1, Reservation::query()->get());

        // assert that we logged the API integration failure
        $this->assertCount(1, ApiNotification::query()->get());
        $notification = ApiNotification::query()->first();
        $this->assertEquals('401', $notification->data['status_code']);
    }

    public function testAvailableVehicle()
    {
        Http::fake([
            "https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials"
                => Http::response($this->tokenResponse()),
            "https://api.eu.keyloop.io/buzz/*"
            => Http::response($this->vehicleAvailableResponse()),
            "*" => Http::response([], 400),
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-id',
            'value' => 'test-client-id',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-secret',
            'value' => 'test-client-secret',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enterprise-id',
            'value' => 'buzz',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => false,
            'is_sold' => false,
            'registration_number' => 'ABC 123',
            'vrm_condensed' => 'ABC123',
        ]);

        $vehicle_attribute = VehicleAttribute::factory()->create([
            'slug' => 'keyloop-id',
        ]);

        VehicleAttributeValue::factory()->create([
            'owner_type' => 'vehicle',
            'owner_id' => $vehicle->id,
            'slug' => $vehicle_attribute->slug,
            'value' => 'bar',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(201);

        $expected_url = 'https://api.eu.keyloop.io/buzz/foo/v1/inventory-vehicles/bar';

        Http::assertSent(function (Request $request) use ($expected_url) {
            if (Str::contains($request->url(), 'inventory')) {
                return $request->url() == $expected_url;
            }

            return false;
        });

        $vehicle->refresh();
        $this->assertFalse($vehicle->is_reserved);
        $this->assertFalse($vehicle->is_sold);

        $notifications = ApiNotification::all();
        $this->assertCount(1, $notifications);
        $this->assertEquals($expected_url, $notifications->first()->data['url']);
        $this->assertEquals('200', $notifications->first()->data['status_code']);
        $this->assertEquals($vehicle->vrm_condensed, $notifications->first()->reference);
    }

    public function testExternallyReservedVehicle()
    {
        Http::fake([
            "https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials"
            => Http::response($this->tokenResponse()),
            "https://api.eu.keyloop.io/buzz/*"
            => Http::response($this->vehicleUnavailableResponse()),
            "*" => Http::response([], 400),
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-id',
            'value' => 'test-client-id',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-secret',
            'value' => 'test-client-secret',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enterprise-id',
            'value' => 'buzz',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => false,
            'is_sold' => false,
        ]);

        $vehicle_attribute = VehicleAttribute::factory()->create([
            'slug' => 'keyloop-id',
        ]);

        VehicleAttributeValue::factory()->create([
            'owner_type' => 'vehicle',
            'owner_id' => $vehicle->id,
            'slug' => $vehicle_attribute->slug,
            'value' => 'bar',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(403);

        Http::assertSent(function (Request $request) {
            if (Str::contains($request->url(), 'inventory')) {
                return $request->url() == 'https://api.eu.keyloop.io/buzz/foo/v1/inventory-vehicles/bar';
            }

            return false;
        });

        $vehicle->refresh();
        $this->assertFalse($vehicle->is_reserved);
        $this->assertFalse($vehicle->is_sold);
    }

    public function testNoApiCallForReservedVehicle()
    {
        Http::fake([
            "https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials"
            => Http::response($this->tokenResponse()),
            "https://api.eu.keyloop.io/buzz/*"
            => Http::response($this->vehicleUnavailableResponse()),
            "*" => Http::response([], 400),
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-id',
            'value' => 'test-client-id',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-secret',
            'value' => 'test-client-secret',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enterprise-id',
            'value' => 'buzz',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => true,
            'is_sold' => false,
        ]);

        $vehicle_attribute = VehicleAttribute::factory()->create([
            'slug' => 'keyloop-id',
        ]);

        VehicleAttributeValue::factory()->create([
            'owner_type' => 'vehicle',
            'owner_id' => $vehicle->id,
            'slug' => $vehicle_attribute->slug,
            'value' => 'bar',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(403);

        Http::assertNothingSent();

        $vehicle->refresh();
        $this->assertTrue($vehicle->is_reserved);
        $this->assertFalse($vehicle->is_sold);
    }

    public function testExternallyReservedVehicleWithUpdate()
    {
        Http::fake([
            "https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials"
            => Http::response($this->tokenResponse()),
            "https://api.eu.keyloop.io/buzz/*"
            => Http::response($this->vehicleUnavailableResponse()),
            "*" => Http::response([], 400),
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'app-details-currency',
            'value' => 'GBC',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'sales-stripe-public_key',
            'value' => 'foo',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-update-local-status',
            'value' => true,
            'type' => 'bool',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-id',
            'value' => 'test-client-id',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-client-secret',
            'value' => 'test-client-secret',
            'type' => 'string',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'keyloop-reservations-enterprise-id',
            'value' => 'buzz',
            'type' => 'string',
        ]);

        $dealership = Dealership::factory()->create([
            'data' => [
                'keyloop-dealer-id' => 'foo',
            ],
        ]);

        $vehicle = Vehicle::factory()->create([
            'dealership_id' => $dealership->id,
            'is_reserved' => false,
            'is_sold' => false,
        ]);

        $vehicle_attribute = VehicleAttribute::factory()->create([
            'slug' => 'keyloop-id',
        ]);

        VehicleAttributeValue::factory()->create([
            'owner_type' => 'vehicle',
            'owner_id' => $vehicle->id,
            'slug' => $vehicle_attribute->slug,
            'value' => 'bar',
        ]);

        $response = $this->asTenant(tenant())
            ->postJson('frontend/reservations/' . $vehicle->slug, [
                'name' => 'test',
                'email' => 'test@test.com',
            ]);

        $response->assertStatus(403);

        Http::assertSent(function (Request $request) {
            if (Str::contains($request->url(), 'inventory')) {
                return $request->url() == 'https://api.eu.keyloop.io/buzz/foo/v1/inventory-vehicles/bar';
            }

            return false;
        });

        $vehicle->refresh();
        $this->assertTrue($vehicle->is_reserved);
        $this->assertFalse($vehicle->is_sold);
    }

    private function tokenResponse()
    {
        return json_encode([
            'access_token' => 'foo123',
        ]);
    }

    private function vehicleUnavailableResponse()
    {
        return str_replace('"AVAILABLE"', '"RESERVED"', $this->vehicleAvailableResponse());
    }

    private function vehicleAvailableResponse()
    {
        return '{
  "vehicleId": "00000000-0000-0000-0000-000000000000",
  "status": "AVAILABLE",
  "inventoryType": "USED",
  "identification": {
    "commissionNumber": "Comm223",
    "stockReference": "srk120",
    "vin": "1FTHF25H1JNB79097",
    "chassis": "68060",
    "licensePlate": "TOY-123",
    "engineNumber": "12345678"
  },
  "description": {
    "class": "CAR",
    "make": "DACIA",
    "model": "SANDERO",
    "variant": "TCE 90",
    "modelYear": 2019,
    "vehicle": "This is a great car",
    "exteriorColor": "Metallic Red",
    "interiorColor": "Black Leather"
  },
  "previousUsage": {
    "code": "code01",
    "description": "code model"
  },
  "referenceData": [
    {
      "providerId": "DMS",
      "makeCode": "SA",
      "modelCode": "SAMPLE",
      "variantCode": "123560",
      "vehicleCode": "ABC12345656",
      "serviceGroupCode": "S2251-10",
      "exteriorColorCode": "C12345",
      "interiorColorCode": "T12345"
    }
  ],
  "specification": {
    "primaryExteriorColor": {
      "code": "RED01",
      "description": "Maroon Red",
      "colorType": "METALLIC",
      "prices": {
        "costPrice": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "listPrice": {
          "netValue": 100,
          "discount": 10,
          "grossValue": 120,
          "taxValue": 150,
          "taxRate": 30,
          "carTax": 10,
          "currencyCode": "GBP"
        }
      }
    },
    "secondaryExteriorColor": {
      "code": "RED01",
      "description": "Maroon Red",
      "colorType": "METALLIC",
      "prices": {
        "costPrice": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "listPrice": {
          "netValue": 100,
          "discount": 10,
          "grossValue": 120,
          "taxValue": 150,
          "taxRate": 30,
          "carTax": 10,
          "currencyCode": "GBP"
        }
      }
    },
    "primaryInteriorColor": {
      "code": "BLACK01",
      "description": "Black",
      "colorType": "OTHER",
      "prices": {
        "costPrice": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "listPrice": {
          "netValue": 100,
          "discount": 10,
          "grossValue": 120,
          "taxValue": 150,
          "taxRate": 30,
          "carTax": 10,
          "currencyCode": "GBP"
        }
      }
    },
    "secondaryInteriorColor": {
      "code": "BLACK01",
      "description": "Black",
      "colorType": "OTHER",
      "prices": {
        "costPrice": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "listPrice": {
          "netValue": 100,
          "discount": 10,
          "grossValue": 120,
          "taxValue": 150,
          "taxRate": 30,
          "carTax": 10,
          "currencyCode": "GBP"
        }
      }
    },
    "optionPacks": [
      {
        "code": "LEATHER",
        "description": "Leather Trim",
        "type": "MANUFACTURERSTANDARD",
        "packPrices": {
          "costPrice": {
            "netValue": 100,
            "grossValue": 120,
            "taxValue": 20,
            "taxRate": 19.99,
            "currencyCode": "GBP"
          },
          "listPrice": {
            "netValue": 100,
            "discount": 10,
            "grossValue": 120,
            "taxValue": 150,
            "taxRate": 30,
            "carTax": 10,
            "currencyCode": "GBP"
          }
        },
        "options": [
          {
            "type": "MANUFACTURERSTANDARD",
            "code": "OPT1",
            "description": "Sample option 1",
            "optionPrices": {
              "costPrice": {
                "netValue": 100,
                "grossValue": 120,
                "taxValue": 20,
                "taxRate": 19.99,
                "currencyCode": "GBP"
              },
              "listPrice": {
                "netValue": 100,
                "discount": 10,
                "grossValue": 120,
                "taxValue": 150,
                "taxRate": 30,
                "carTax": 10,
                "currencyCode": "GBP"
              }
            }
          }
        ]
      }
    ],
    "options": [
      {
        "type": "MANUFACTURERSTANDARD",
        "code": "OPT1",
        "description": "Sample option 1",
        "optionPrices": {
          "costPrice": {
            "netValue": 100,
            "grossValue": 120,
            "taxValue": 20,
            "taxRate": 19.99,
            "currencyCode": "GBP"
          },
          "listPrice": {
            "netValue": 100,
            "discount": 10,
            "grossValue": 120,
            "taxValue": 150,
            "taxRate": 30,
            "carTax": 10,
            "currencyCode": "GBP"
          }
        }
      }
    ]
  },
  "mileage": {
    "unit": "KM",
    "value": 120000
  },
  "dates": {
    "registrationDate": "2021-07-16T00:00:00.000Z",
    "firstRegistrationDate": "2021-07-16T00:00:00.000Z"
  },
  "prices": {
    "listPrice": {
      "netValue": 100,
      "discount": 10,
      "grossValue": 120,
      "taxValue": 150,
      "taxRate": 30,
      "carTax": 10,
      "currencyCode": "GBP"
    },
    "costPrice": {
      "netValue": 100,
      "discount": 10,
      "grossValue": 120,
      "taxValue": 150,
      "taxRate": 30,
      "carTax": 10,
      "currencyCode": "GBP"
    },
    "deliveryPrice": {
      "netValue": 100,
      "grossValue": 120,
      "taxValue": 20,
      "taxRate": 19.99,
      "currencyCode": "GBP"
    },
    "nonRecoverableCosts": [
      {
        "actualCost": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "estimatedCost": {
          "netValue": 100,
          "grossValue": 120,
          "taxValue": 20,
          "taxRate": 19.99,
          "currencyCode": "GBP"
        },
        "code": "COST2",
        "description": "Sample Cost"
      }
    ],
    "carTax": {
      "startDate": "2021-07-16T00:00:00.000Z",
      "type": "string",
      "carTaxNew": {
        "carTax": 10,
        "currencyCode": "GBP"
      }
    },
    "AdjustedStockValue": {
      "value": 10,
      "currencyCode": "GBP"
    }
  },
  "isVatQualifyingVehicle": true,
  "inventoryDates": {
    "orderDate": "2021-07-16T00:00:00.000Z",
    "buildDate": "2021-07-16T00:00:00.000Z",
    "estimatedReceivedDate": "2021-07-16T00:00:00.000Z",
    "receivedDate": "2021-07-16T00:00:00.000Z",
    "adoptedDate": "2021-07-16T00:00:00.000Z",
    "consignedDate": "2021-07-16T00:00:00.000Z",
    "availableForSaleDate": "2021-07-16T00:00:00.000Z"
  },
  "reservation": {
    "reservationDate": "2021-07-16T00:00:00.000Z",
    "reservedUntilDate": "2021-07-16T00:00:00.000Z",
    "reservedBy": {
      "salesPersonId": "12345",
      "name": "James Doe"
    }
  },
  "links": [
    {
      "method": "GET",
      "rel": "self",
      "href": "http//example.com/api/d3de97a4-f061-4f15-a228-2b4355148fdb",
      "title": "Vehicle Inventory Links"
    }
  ]
}';
    }
}
