<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\FakeCampaignMonitor;
use Tests\FakeMailchimp;
use Tests\TestCase;

class NewsletterControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testSubmitMailchimpNew()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'newsletters-mailchimp-enabled',
            'value' => true,
            'type' => 'bool',
        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test@example.com'
            ]);

        $response->assertStatus(200);
        $this->assertTrue($response->json('success'));
    }

    public function testSubmitMailchimpAlreadyExists()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'newsletters-mailchimp-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        FakeMailchimp::$return_subscribed = true;
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test@example.com'
            ]);

        $response->assertStatus(200);
        $this->assertTrue($response->json('success'));
    }

    public function testSubmitCampaignMonitor()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'newsletters-campaign-monitor-enabled',
            'value' => true,
            'type' => 'bool',
        ]);
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test@example.com'
            ]);

        $response->assertStatus(200);
        $this->assertTrue($response->json('success'));
    }

    public function testSubmitCampaignMonitorAlreadyExists()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'newsletters-campaign-monitor-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        FakeCampaignMonitor::$return_subscribed = true;
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test@example.com'
            ]);

        $response->assertStatus(200);
        $this->assertTrue($response->json('success'));
    }

    public function testSubmitCampaignMonitorBadRequest()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'newsletters-campaign-monitor-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        FakeCampaignMonitor::$allow_signing_up = false;
        FakeCampaignMonitor::$return_subscribed = false;
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test@example.com'
            ]);

        $response->assertStatus(200);
        $this->assertFalse($response->json('success'));
    }

    public function testSubmitBadRequest()
    {
        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), []);

        $response->assertStatus(422);
        $this->assertArrayHasKey('email', $response->json('errors'));

        $response = $this->asTenant(tenant())
            ->postJson(route('newsletter-signup'), [
                'email' => 'test-me'
            ]);

        $response->assertStatus(422);
        $this->assertArrayHasKey('email', $response->json('errors'));
    }
}
