<?php

namespace App;

use App\Http\Requests\SubmitEnquiryRequest;
use Illuminate\Support\Facades\Log;
use Mtc\Crm\Models\Enquiry;

class EnquiryFormHandler
{
    /**
     * Handle incoming Request
     *
     * @param SubmitEnquiryRequest $request
     * @return array|string[]
     */
    public function handle(SubmitEnquiryRequest $request): array
    {
        if ($this->processRequest($request)) {
            return [
                'status' => 'ok',
                'message' => $this->getSuccessMessage($request),

            ];
        }
        return [
            'status' => 'error',
            'message' => $this->getErrorMessage()
        ];
    }

    /**
     * Process enquiry request to record
     * @param SubmitEnquiryRequest $request
     * @return bool
     */
    protected function processRequest(SubmitEnquiryRequest $request): bool
    {
        try {
            Enquiry::query()
                ->create([
                    'form_id' => $request->input('form_id'),
                    'email' => $this->getEmailFromRequest($request),
                    'type_id' => $request->form()->type_id,
                    'status_id' => $request->form()->initial_status_id,
                    'title' => $this->getTitleFromRequest($request),
                    'message' => '',
                    'details' => $request->input('questions'),
                    'data' => [
                        'questions' => $request->form()?->questions()->pluck('data', 'id'),
                    ],
                    'assigned_user_id' => $request->form()->initial_assignee,
                ]);
        } catch (\Exception $exception) {
            Log::error('Failed to process Enquiry Submission', [
                'input' => $request->input(),
                'error' => $exception->getMessage(),
            ]);
            return false;
        }
        return true;
    }

    /**
     * Success message
     *
     * @param SubmitEnquiryRequest $request
     * @return string
     */
    protected function getSuccessMessage(SubmitEnquiryRequest $request): string
    {
        return $request->form()->success_message ?? '';
    }

    /**
     * Error message
     *
     * @return string
     */
    protected function getErrorMessage(): string
    {
        return 'Failed to submit your enquiry. Please make sure all required fields are filled and try again';
    }

    /**
     * Get the enquiry title/subject from form fields and request data
     *
     * @param SubmitEnquiryRequest $request
     * @return string
     */
    protected function getTitleFromRequest(SubmitEnquiryRequest $request): string
    {
        return $request->form()
            ->questions
            ->filter(fn($question) => $question->is_enquiry_subject)
            ->sortBy(fn($question) => $question->order)
            ->map(fn($question) => $request->input('questions.' . $question->id . '.value'))
            ->implode(' ');
    }

    /**
     * Find email field value based off the form fields and request
     *
     * @param SubmitEnquiryRequest $request
     * @return string|null
     */
    protected function getEmailFromRequest(SubmitEnquiryRequest $request): string|null
    {
        $question_id = $request->form()
            ->questions
            ->filter(fn($question) => $question->is_customers_email)
            ->first()
            ->id ?? null;

        return $question_id ? $request->input('questions.' . $question_id . '.value') : '';
    }
}
