<?php

namespace App;

use App\Facades\Settings;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Filter\Contracts\IsFilter;
use App\Seo\FilterSeoDataMap;
use Mtc\Filter\FilterIndex;

class Filter extends \Mtc\Filter\Filter
{
    use HasSeoData;

    /**
     * Find the id of the slug
     *
     * @param string $model_type
     * @param $slug
     * @return int
     */
    public static function matchSlugToId(string $model_type, $slug): int
    {
        return FilterIndex::query()
            ->where('filter_type', $model_type)
            ->where('slug', $slug)
            ->firstOrFail()
            ->filter_id;
    }

    /**
     * Parse Request to selections and filter data
     *
     * @return array
     */
    public function parseRequest(): array
    {
        $this->decodeRequest();

        return [
            'seo' => $this->getSeoData($this->request->path(), 'filter', (new FilterSeoDataMap($this))),
            'selections' => $this->getSelectionList(),
            'sort_by' => $this->active_sort_option_name,
            'base_url' => url($this->config['url_entrypoint']),
            'page' => $this->request->input('page', 1),
        ];
    }

    /**
     * Handle Ajax request, perform filtering
     *
     * @return array
     */
    public function handle(): array
    {
        $this->request = request();
        $this->selections = $this->groupSelectionsByType($this->request->input('selections', []));
        $this->run();

        return [
            'seo' => $this->getSeoData($this->request->path(), 'filter', (new FilterSeoDataMap($this))),
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'sort_options' => $this->getSortOptions(),
        ];
    }

    /**
     * Get enabled filters
     *
     * @return Collection
     */
    public function getFilters(): Collection
    {
        if (empty($this->filters)) {
            $this->filters = collect($this->config['filters'])
                ->map(fn($filter_class) => App::make($filter_class))
                ->filter(fn($filter) => $filter instanceof IsFilter)
                ->filter(fn($filter, $name) => Settings::get("automotive.vehicle-filters.$name") === true);
        }

        return $this->filters;
    }

    /**
     * Fetch the list of supported sort options
     * Returned in format of slug => display name
     *
     * @return array
     */
    public function getSortOptions(): array
    {
        return collect($this->config['sort_options'])
            ->filter(fn($filter, $filter_name) => Settings::get("automotive.vehicle-sorting.$filter_name") === true)
            ->map(fn($sort_class, $sort_name) => __("filter::filter.sort_options.$sort_name"))
            ->toArray();
    }
}
