<?php

namespace App\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\MercuryDataModels\Vehicle;

class PriceFilter extends IsFilter implements CustomPatternFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->when(
            $selection[0]['payment_type'] === 'monthly',
            function ($query) use ($selection) {
                $query->where('monthly_price', '>=', $selection[0]['min'] ?? 0)
                    ->where('monthly_price', '<', $selection[0]['max'] ?? PHP_INT_MAX);
            },
            function ($query) use ($selection) {
                $query->where('price', '>=', $selection[0]['min'] ?? 0)
                    ->where('price', '<', $selection[0]['max'] ?? PHP_INT_MAX);
            }
        );
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([
            'min' => floor(Vehicle::query()->where($product_filtering)->min('price')),
            'max' => ceil(Vehicle::query()->where($product_filtering)->max('price')),
        ]);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Vehicle::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.price');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
        ];
    }

    /**
     * Specify UI component type used for this filter
     *
     * @return string
     */
    public function uiComponentType(): string
    {
        return 'range-slider-filter';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/price-from-[0-9]+-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/price-from-([0-9]+)-to-([0-9]+)/', $selection, $matches);
        return [
            'min' => $matches[1],
            'max' => $matches[2],
        ];
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return __('filter::filter.price_range_term_name', ['min' => $selection['min'], 'max' => $selection['max']]);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'price-from-' . $selection['min'] . '-to-' . $selection['max'];
    }

    /**
     * Specify whether model should be re-indexed in the index
     * @return bool
     */
    public function reindexModel(): bool
    {
        return false;
    }
}
